/* 
    TowBowlTactics: Strategy game for X11

    Copyright (C) 2001 Pascal Bourut (toweld@rocketmail.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "DialogBox.h"
#include "SFont.h"
#include "debug.h"

DialogBox::DialogBox(SDL_Surface *screen){
	DialogBox(screen, "En");
	d2 = new Dice(D2);
}

DialogBox::DialogBox(SDL_Surface *screen,const char *lg){
	this->screen = screen;

	SDL_Surface *tmp;
	tmp = SDL_AllocSurface(SDL_HWSURFACE, 
							screen->w,screen->h,
							screen->format->BitsPerPixel,
							screen->format->Rmask,
							screen->format->Gmask,
							screen->format->Bmask,
							screen->format->Amask);
	old_screen = SDL_DisplayFormat(tmp);
	SDL_FreeSurface(tmp);

	tmp = SDL_AllocSurface(SDL_HWSURFACE, 
							DIALOG_W,DIALOG_H,
							screen->format->BitsPerPixel,
							screen->format->Rmask,
							screen->format->Gmask,
							screen->format->Bmask,
							screen->format->Amask);

	dialog_box_surf = SDL_DisplayFormat(tmp);
	SDL_FreeSurface(tmp);

	type = DIALOG_NONE;
	visible = false;
	translator = new Translator(lg);
	d2 = new Dice(D2);
}

DialogBox::~DialogBox(){
	free_images();
	delete translator;
	delete d2;
	SDL_FreeSurface(old_screen);
	SDL_FreeSurface(dialog_box_surf);
}

void DialogBox::load_images(void){
  dialog_img[DIALOG_YESNO] = load_image(ADD_IMG_PATH("dialog/dialog_yesno.png"),TRANSPARENT);
  dialog_img[DIALOG_BLOCK] = load_image(ADD_IMG_PATH("dialog/dialog_block.png"),TRANSPARENT); 
  dialog_img[DIALOG_INFO] = load_image(ADD_IMG_PATH("dialog/dialog_info.png"),TRANSPARENT);
  dialog_img[DIALOG_TOSS] = load_image(ADD_IMG_PATH("dialog/dialog_toss.png"),TRANSPARENT);
  dialog_img[DIALOG_CHOICE] = load_image(ADD_IMG_PATH("dialog/dialog_choice.png"),TRANSPARENT);
  close_img = load_image(ADD_IMG_PATH("dialog/close.jpg"),NO_TRANSPARENT);
  yes_img = load_image(ADD_IMG_PATH("dialog/yes.jpg"),NO_TRANSPARENT);
  no_img = load_image(ADD_IMG_PATH("dialog/no.jpg"),NO_TRANSPARENT);
  a_img = load_image(ADD_IMG_PATH("dialog/a.jpg"),NO_TRANSPARENT);
  b_img = load_image(ADD_IMG_PATH("dialog/b.jpg"),NO_TRANSPARENT);
  indicator_img = load_image(ADD_IMG_PATH("dialog/indicator.png"),TRANSPARENT);
  block_dices_img = load_image(ADD_IMG_PATH("dialog/block_dices.jpg"),NO_TRANSPARENT);
  icons_img = load_image(ADD_IMG_PATH("dialog/icons.jpg"),NO_TRANSPARENT);
  defender_img = load_image(ADD_IMG_PATH("dialog/defender.jpg"),NO_TRANSPARENT);
  reroll_img = load_image(ADD_IMG_PATH("dialog/reroll.jpg"),NO_TRANSPARENT);
  mask_img = load_image(ADD_IMG_PATH("dialog/mask.png"),NO_TRANSPARENT);
  pieces_img = load_image(ADD_IMG_PATH("dialog/pieces.jpg"),NO_TRANSPARENT);
  toss_bottom_img = load_image(ADD_IMG_PATH("dialog/toss_choice.jpg"),NO_TRANSPARENT);
  kick_receive_img = load_image(ADD_IMG_PATH("dialog/toss_buttons.jpg"),NO_TRANSPARENT);
  not_agree_img = load_image(ADD_IMG_PATH("dialog/not_agree.jpg"),NO_TRANSPARENT);
  head_tail_img = load_image(ADD_IMG_PATH("dialog/head_tail.jpg"),NO_TRANSPARENT);
  font_img = load_image(ADD_IMG_PATH("dialog/font.png"),TRANSPARENT);
  SDL_SetAlpha(mask_img, MASK_ALPHA_FLAGS, MASK_ALPHA);

  InitFont(font_img);
}

void DialogBox::free_images(void){
	if(dialog_img[DIALOG_YESNO]) SDL_FreeSurface(dialog_img[DIALOG_YESNO]);
	if(dialog_img[DIALOG_BLOCK]) SDL_FreeSurface(dialog_img[DIALOG_BLOCK]);
	if(dialog_img[DIALOG_INFO]) SDL_FreeSurface(dialog_img[DIALOG_INFO]);
	if(dialog_img[DIALOG_TOSS]) SDL_FreeSurface(dialog_img[DIALOG_TOSS]);
	if(dialog_img[DIALOG_CHOICE]) SDL_FreeSurface(dialog_img[DIALOG_CHOICE]);
	if(close_img) SDL_FreeSurface(close_img);
	if(yes_img) SDL_FreeSurface(yes_img);
	if(no_img) SDL_FreeSurface(no_img);
	if(a_img) SDL_FreeSurface(a_img);
	if(b_img) SDL_FreeSurface(b_img);
	if(indicator_img) SDL_FreeSurface(indicator_img);
	if(block_dices_img) SDL_FreeSurface(block_dices_img);
	if(icons_img) SDL_FreeSurface(icons_img);
	if(defender_img) SDL_FreeSurface(defender_img);
	if(reroll_img) SDL_FreeSurface(reroll_img);
	if(mask_img) SDL_FreeSurface(mask_img);
	if(pieces_img) SDL_FreeSurface(pieces_img);
	if(toss_bottom_img) SDL_FreeSurface(toss_bottom_img);
	if(kick_receive_img) SDL_FreeSurface(kick_receive_img);
	if(not_agree_img) SDL_FreeSurface(not_agree_img);
	if(head_tail_img) SDL_FreeSurface(head_tail_img);
	if(font_img) SDL_FreeSurface(font_img);
}

int DialogBox::open_toss(SDL_Surface* screen,const char* home,const char* visitor){
	bool done=false;
	int r;
	type = DIALOG_TOSS;
 
	this->title = strdup(home);

	team_name[HOME] = strdup(home);
	team_name[VISITOR] = strdup(visitor);

	init_graphics(screen);

	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}

	visible = false;
	this->title[0] = '\0';

	return r; 
}

int DialogBox::open_yesno(SDL_Surface* screen,int icons_id,int msg_id, const char* title){
	bool done=false;
	int r;
	type = DIALOG_YESNO;

	this->icons_id = icons_id;
	this->title = strdup(title);

	this->msg = strdup(translator->get_text_by_ID(msg_id));

	init_graphics(screen);

	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}

	visible = false;
	this->title[0] = '\0';

	return r; 
}


int DialogBox::open_yesno_s(SDL_Surface* screen,int icons_id,int msg_id, const char* title,const char* param_s){
	bool done=false;
	int r;
	type = DIALOG_YESNO;

	this->icons_id = icons_id;
	this->title = strdup(title);

	char * temp;
	temp = (char*)malloc( CHAR_256 );

	const char * str = translator->get_text_by_ID(msg_id);

	sprintf(temp, str, param_s);
	
	this->msg = strdup(temp);

	free(temp);

	init_graphics(screen);

	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}

	visible = false;
	this->title[0] = '\0';

	return r; 
}

int DialogBox::open_choice(SDL_Surface* screen,int icons_id,int msg_id, const char* title){
	bool done=false;
	int r;
	type = DIALOG_CHOICE;
	this->icons_id = icons_id;
	this->title = strdup(title);
	this->msg = strdup(translator->get_text_by_ID(msg_id));
	init_graphics(screen);
  
	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}
	visible = false;
	this->title[0] = '\0';
	return r; 
}


int DialogBox::open_choice_s(SDL_Surface* screen,int icons_id,int msg_id, const char* title,  const char* param_s){
	bool done=false;
	int r;
	type = DIALOG_CHOICE;
	this->icons_id = icons_id;
	this->title = strdup(title);
    
	char * temp;
	temp = (char*)malloc(CHAR_256);
	const char * str = translator->get_text_by_ID(msg_id);
		
	sprintf(temp, str, param_s);
	this->msg = strdup(temp);

	free(temp);

	init_graphics(screen);
  
	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}
	visible = false;
	this->title[0] = '\0';
	return r; 
}

int DialogBox::open_block(SDL_Surface* screen,int* dices,int who_choose,bool can_reroll,const char* title){

	bool done=false;
	int r;

	type = DIALOG_BLOCK;
	this->who_choose = who_choose;
	this->can_reroll = can_reroll;
	this->dices = dices;

	this->title = strdup(title);
  
	init_graphics(screen);

	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}

	visible = false;
	this->title[0] = '\0';

	return r;
}

int DialogBox::open_info(SDL_Surface *screen,int icons_id,int msg_id,bool can_complain, const char* title){
	
	bool done=false;
	int r;

	type = DIALOG_INFO;
	this->icons_id = icons_id; 
	this->title = strdup(title); 

	this->can_complain = can_complain;	
	this->msg = strdup(translator->get_text_by_ID(msg_id));

	init_graphics(screen);

	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}

	visible = false;
	this->title[0] = '\0';

	return r;
}

int DialogBox::open_info_s(SDL_Surface *screen,int icons_id,int msg_id,bool can_complain, const char* title, const char* param_s){
	
	bool done=false;
	int r;

	type = DIALOG_INFO;
	this->icons_id = icons_id; 
	this->title = strdup(title); 

	this->can_complain = can_complain;
		
	char * temp;
	temp = (char*)malloc(CHAR_256);

	const char * str = translator->get_text_by_ID(msg_id);

	sprintf(temp, str, param_s);
	this->msg = strdup(temp);

	free(temp);

	init_graphics(screen);

	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}

	visible = false;
	this->title[0] = '\0';

	return r;
}

int DialogBox::open_info_si(SDL_Surface *screen,int icons_id,int msg_id,bool can_complain, const char* title, const char* param_s, int param_i){
	
	bool done=false;
	int r;

	type = DIALOG_INFO;
	this->icons_id = icons_id; 
	this->title = strdup(title); 

	this->can_complain = can_complain;
  
	char * temp;
	temp = (char*)malloc(CHAR_256);

	const char * str = translator->get_text_by_ID(msg_id);

	sprintf(temp, str, param_s, param_i);
	this->msg = strdup(temp);

	free(temp);

	init_graphics(screen);

	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}

	visible = false;
	this->title[0] = '\0';

	return r;
}

int DialogBox::open_info_i(SDL_Surface *screen,int icons_id,int msg_id,bool can_complain, const char* title, int param_i){
	bool done=false;
	int r;
  
	type = DIALOG_INFO;
	this->icons_id = icons_id; 
	this->title = strdup(title); 
	this->can_complain = can_complain;
	char * temp;
	temp = (char*)malloc(CHAR_256);

	const char * str = translator->get_text_by_ID(msg_id);
	sprintf(temp, str,param_i);
	this->msg = strdup(temp);

	free(temp);

	init_graphics(screen);
	while(!done){
		draw();
		r = handle_event();
		done = (r!=NOP);   
	}
	
	visible = false;
	this->title[0] = '\0';

	return r;
}


void DialogBox::init_graphics(SDL_Surface *screen){
  visible = true;
  x = (screen->w - DIALOG_W)/2; 
  y = (screen->h - DIALOG_H)/2; 
  BlitSurface(screen, NULL, old_screen, NULL);
  BlitSurface(mask_img,NULL,old_screen,NULL);
  BlitSurface(old_screen,NULL,screen,NULL);
  over_close = false;
  over_yes = false;
  over_no = false;
  over_a = false;
  over_b = false;
  over_reroll = false;
  move_dialog = false;
  over_kick = false;
  over_receive = false;
  over_head = false;
  over_tail = false;
  over_not_agree = false;
  face_choice = -1;
  face_result = -1;
  toss_state = TOSS_CHOICE;
  dices_indicator_x_dest = -1;
  dices_indicator_x_current = DICE_2_INDICATOR_X;
  pieces_indicator_x = (PIECE_HEAD_INDICATOR_X+PIECE_TAIL_INDICATOR_X)/2;
  pieces_toss_count = 0;
  update = true;
}

int DialogBox::handle_event(){
  
  SDL_Event event;
  static int old_x = 0;
  static int old_y = 0;
  
  while(SDL_PollEvent(&event)) { 
    switch (event.type) {
    case SDL_KEYDOWN:
      switch(event.key.keysym.sym){
	  case SDLK_a:
	  case SDLK_z:
		  if(type==DIALOG_CHOICE){
			  update = true;
			  return CHOICE_A; 
		  }
		  break;
	  case SDLK_b:
		  if(type==DIALOG_CHOICE){
			  update = true;
			  return CHOICE_B; 
		  }
		  break;
	  case SDLK_o:
	  case SDLK_y:
		  if(type==DIALOG_YESNO){
			  update = true;
			  return YES; 
		  }
		  break;
	  case SDLK_n:
		  if(type==DIALOG_YESNO){
			  update = true;
			  return NO; 
		  }
		  break;
	  case SDLK_c:
		  if(type==DIALOG_INFO){
			  update = true;
			  return CLOSE; 
		  }
		  break;
      case SDLK_k:
		  if(type==DIALOG_TOSS && toss_state==TOSS_KICK_OR_RECEIVE) {
			  update = true;
			  return (face_choice==face_result)?HOME_KICK:VISITOR_KICK;
		  }
		  break;
      case SDLK_r:
		  if(type==DIALOG_TOSS && toss_state==TOSS_KICK_OR_RECEIVE) {
			  update = true;
			  return (face_choice==face_result)?VISITOR_KICK:HOME_KICK;
		  }
		  break;
	  case SDLK_h:
		  if(type==DIALOG_TOSS && toss_state==TOSS_CHOICE) {
			  update = true;
			  face_choice = HEAD;
			  toss_state = TOSS_UP;
			  face_result = d2->roll();
		  }
		  break;
	  case SDLK_t:
		  if(type==DIALOG_TOSS && toss_state==TOSS_CHOICE) {
			  update = true;
			  face_choice = TAIL;
			  toss_state = TOSS_UP;
			  face_result = d2->roll();
		  }
		  break;
	  case SDLK_RETURN:
	update = true;
	switch(type){
	case DIALOG_TOSS:
	  switch(toss_state){
	  case TOSS_CHOICE:
	    if(over_head) {
	      face_choice = HEAD;
	      toss_state = TOSS_UP;
	      face_result = d2->roll();
	    }
	    else if (over_tail) {
	      face_choice = TAIL;
	      toss_state = TOSS_UP;
	      face_result = d2->roll();
	    }
	    break;
	  case TOSS_KICK_OR_RECEIVE:
	    if(over_kick) 
	      return (face_choice==face_result)?HOME_KICK:VISITOR_KICK;
	    else if(over_receive) 
	      return (face_choice==face_result)?VISITOR_KICK:HOME_KICK;
	    break;
	  }
	  break; 
	case DIALOG_INFO:
	  if(over_not_agree) return NOT_AGREE;
	  return CLOSE;
	  break;
	case DIALOG_YESNO:
	  if(over_no) return NO;
	  if(over_yes) return YES;
	  break;
	case DIALOG_CHOICE:
	  if(over_a) return CHOICE_A;
	  if(over_b) return CHOICE_B;
	  break;
	case DIALOG_BLOCK:
	  if(over_reroll) return REROLL;
	  
	  switch(dices_indicator_x_dest){
	  case DICE_1_INDICATOR_X:
	    return FIRST_DICE;
	    break;
	  case DICE_2_INDICATOR_X:
	    return SECOND_DICE;
	    break;    
	  case DICE_3_INDICATOR_X:
	    return THIRD_DICE;
	    break;
	  }
	  break;
	}
	break;
      case SDLK_ESCAPE:
	update = true;
	switch(type){
	case DIALOG_INFO:
	  return CLOSE;
	  break;
	}
	break; 
      case SDLK_LEFT:
	update = true;
	switch(type){
	case DIALOG_INFO:
	  if(!over_close) SDL_WarpMouse(CLOSE_BUTTON_X+CLOSE_BUTTON_W/2+x,CLOSE_BUTTON_Y+CLOSE_BUTTON_H+y-1); 
	  break;
	case DIALOG_TOSS:
	  switch(toss_state){
	  case TOSS_CHOICE:
	    if(!over_head) SDL_WarpMouse(HEAD_BUTTON_X+HEAD_BUTTON_W/2+x,HEAD_BUTTON_Y+HEAD_BUTTON_H+y-1); 
	    break;
	  case TOSS_KICK_OR_RECEIVE:
	    if(!over_kick) SDL_WarpMouse(KICK_BUTTON_X+KICK_BUTTON_W/2+x,KICK_BUTTON_Y+KICK_BUTTON_H+y-1); 
	    break;
	  }
	  break;
	case DIALOG_YESNO:
	  if(!over_yes) SDL_WarpMouse(YES_BUTTON_X+YES_BUTTON_W/2+x,YES_BUTTON_Y+YES_BUTTON_H+y-1); 
	  break;
	case DIALOG_CHOICE:
	  if(!over_a) SDL_WarpMouse(YES_BUTTON_X+YES_BUTTON_W/2+x,YES_BUTTON_Y+YES_BUTTON_H+y-1); 
	  break;
	case DIALOG_BLOCK:
	  if(!over_reroll && (dices[0]!=-1) && dices_indicator_x_dest == DICE_2_INDICATOR_X){
	    SDL_WarpMouse(DICE_1_BUTTON_X+DICES_BUTTON_W/2+x,DICE_1_BUTTON_Y+DICES_BUTTON_H+y-1); 
	  }
	  else if(!over_reroll && (dices[1]!=-1) && dices_indicator_x_dest == DICE_3_INDICATOR_X){
	    SDL_WarpMouse(DICE_2_BUTTON_X+DICES_BUTTON_W/2+x,DICE_2_BUTTON_Y+DICES_BUTTON_H+y-1); 
	  }
	  else if(!over_reroll && (dices[0]!=-1) && dices_indicator_x_dest == -1) {
	    SDL_WarpMouse(DICE_1_BUTTON_X+DICES_BUTTON_W/2+x,DICE_1_BUTTON_Y+DICES_BUTTON_H+y-1); 
	  }
	  break;
	}
	break;
      case SDLK_RIGHT:
	update = true;
	switch(type){
	case DIALOG_INFO:
	  if(!over_close) SDL_WarpMouse(CLOSE_BUTTON_X+CLOSE_BUTTON_W/2+x,CLOSE_BUTTON_Y+CLOSE_BUTTON_H+y-1); 
	  break; 
	case DIALOG_TOSS:
	  switch(toss_state){
	  case TOSS_CHOICE:
	    if(!over_tail) SDL_WarpMouse(TAIL_BUTTON_X+TAIL_BUTTON_W/2+x,TAIL_BUTTON_Y+TAIL_BUTTON_H+y-1);
	    break;
	  case TOSS_KICK_OR_RECEIVE:
	    if(!over_receive) SDL_WarpMouse(RECEIVE_BUTTON_X+RECEIVE_BUTTON_W/2+x,RECEIVE_BUTTON_Y+RECEIVE_BUTTON_H+y-1); 
	    break;
	  }
	  break;
	case DIALOG_YESNO:
	  if (!over_no) SDL_WarpMouse(NO_BUTTON_X+NO_BUTTON_W/2+x,NO_BUTTON_Y+NO_BUTTON_H+y-1); 
	  break;
	case DIALOG_CHOICE:
	  if (!over_b) SDL_WarpMouse(NO_BUTTON_X+NO_BUTTON_W/2+x,NO_BUTTON_Y+NO_BUTTON_H+y-1); 
	  break;
	case DIALOG_BLOCK:
	  if( !over_reroll && (dices[2]!=-1) && dices_indicator_x_dest == DICE_2_INDICATOR_X){
	    SDL_WarpMouse(DICE_3_BUTTON_X+DICES_BUTTON_W/2+x,DICE_3_BUTTON_Y+DICES_BUTTON_H+y-1); 
	  }
	  else if(!over_reroll && (dices[1]!=-1) && dices_indicator_x_dest == DICE_1_INDICATOR_X){
	    SDL_WarpMouse(DICE_2_BUTTON_X+DICES_BUTTON_W/2+x,DICE_2_BUTTON_Y+DICES_BUTTON_H+y-1);
	  }
	  else if(!over_reroll && (dices[2]!=-1) && dices_indicator_x_dest == -1) {
	    SDL_WarpMouse(DICE_3_BUTTON_X+DICES_BUTTON_W/2+x,DICE_3_BUTTON_Y+DICES_BUTTON_H+y-1); 
	  }
	  else if(!over_reroll && (dices[1]!=-1) && dices_indicator_x_dest == -1) {
	    SDL_WarpMouse(DICE_2_BUTTON_X+DICES_BUTTON_W/2+x,DICE_2_BUTTON_Y+DICES_BUTTON_H+y-1); 
	  }
	  else if(!over_reroll && (dices[0]!=-1) && dices_indicator_x_dest == -1) {
	    SDL_WarpMouse(DICE_1_BUTTON_X+DICES_BUTTON_W/2+x,DICE_1_BUTTON_Y+DICES_BUTTON_H+y-1); 
	  }
	  break;
	}
	break;
      case SDLK_UP:
	update = true;
	switch(type){
	case DIALOG_INFO:
	  if(can_complain && !over_not_agree){
	    SDL_WarpMouse(NOT_AGREE_X+NOT_AGREE_W/2+x,NOT_AGREE_Y+NOT_AGREE_H+y-1);
	  }
	  break;
	case DIALOG_BLOCK:
	  if(can_reroll && over_reroll){
	    switch(dices_indicator_x_dest){
	    case DICE_1_INDICATOR_X:
	      SDL_WarpMouse(DICE_1_BUTTON_X+DICES_BUTTON_W/2+x,DICE_1_BUTTON_Y+DICES_BUTTON_H+y-1);
	      break;
	    case DICE_2_INDICATOR_X:
	      SDL_WarpMouse(DICE_2_BUTTON_X+DICES_BUTTON_W/2+x,DICE_2_BUTTON_Y+DICES_BUTTON_H+y-1);
	      break;    
	    case DICE_3_INDICATOR_X:
	      SDL_WarpMouse(DICE_3_BUTTON_X+DICES_BUTTON_W/2+x,DICE_3_BUTTON_Y+DICES_BUTTON_H+y-1);
	      break;
	    default:
	      SDL_WarpMouse(x,y);
	      break;
	    }
	    break; 
	  }
	  break;
	}
	break;
      case SDLK_DOWN:
	update = true;
	switch(type){
	case DIALOG_INFO:
	  if(can_complain && over_not_agree){
	    SDL_WarpMouse(CLOSE_BUTTON_X+CLOSE_BUTTON_W/2+x,CLOSE_BUTTON_Y+CLOSE_BUTTON_H+y-1); 
	  }
	  break;  
	case DIALOG_BLOCK:
	  if(can_reroll && !over_reroll){
	    SDL_WarpMouse(REROLL_BUTTON_X+REROLL_BUTTON_W/2+x,REROLL_BUTTON_Y+REROLL_BUTTON_H+y-1);
	  }
	  break;
	}
	break;
      default:
	break;
      }
      
      break; // esac de SDL_KEYDOWN
    case SDL_MOUSEMOTION:
      if ( OVER_BOX(event.motion.x,event.motion.y) || OVER_BOX(old_x,old_y) ) update = true;
      
      old_x = event.motion.x;
      old_y = event.motion.y;

      if ( move_dialog ){
	x += event.motion.xrel;
	y += event.motion.yrel;
	
	if ( x < 0 ) x = 0;
	else if ( x > (screen->w-DIALOG_W) ) x = screen->w-DIALOG_W;
	if ( y < 0 ) y = 0;
	else if ( y > (screen->h-DIALOG_H) ) y = screen->h-DIALOG_H;
      update = true;
	  }
      else {
	over_close = false;
	over_yes = false;
	over_no = false;
	over_a = false;
	over_b = false;
	over_reroll = false;
	over_head = false;
	over_tail = false;
	over_kick = false;
	over_receive = false;
	over_not_agree = false;
	switch(type){
	case DIALOG_TOSS:
	  switch(toss_state){
	  case TOSS_CHOICE:
	    if(OVER_HEAD_BUTTON(event.motion.x-x,event.motion.y-y) || OVER_HEAD_PIECE(event.motion.x-x,event.motion.y-y) ){
	      over_head = true;
	    } 
	    else if(OVER_TAIL_BUTTON(event.motion.x-x,event.motion.y-y) || OVER_TAIL_PIECE(event.motion.x-x,event.motion.y-y)){
	      over_tail = true;
	    } 
	    break;
	  case TOSS_KICK_OR_RECEIVE:
	    if(OVER_KICK_BUTTON(event.motion.x-x,event.motion.y-y)){
	      over_kick = true;
	    } 
	    else if(OVER_RECEIVE_BUTTON(event.motion.x-x,event.motion.y-y)){
	      over_receive = true;
	    } 
	    break;
	  }
	  break;
	case DIALOG_INFO:
	  if(OVER_CLOSE_BUTTON(event.motion.x-x,event.motion.y-y)){
	    over_close = true;
	  } 
	  else if(OVER_NOT_AGREE(event.motion.x-x,event.motion.y-y) && can_complain){
	    over_not_agree = true;
	  } 
	  break;
	case DIALOG_YESNO:
	  if(OVER_YES_BUTTON(event.motion.x-x,event.motion.y-y)){
	    over_yes = true;
	  } 
	  else if(OVER_NO_BUTTON(event.motion.x-x,event.motion.y-y)){
	    over_no = true;
	  } 
	  break;
	case DIALOG_CHOICE:
	  if(OVER_A_BUTTON(event.motion.x-x,event.motion.y-y)){
	    over_a = true;
	  } 
	  else if(OVER_B_BUTTON(event.motion.x-x,event.motion.y-y)){
	    over_b = true;
	  } 
	  break;
	case DIALOG_BLOCK:
	  if(OVER_DICE_1_BUTTON(event.motion.x-x,event.motion.y-y) && (dices[0]!=-1)){
	    dices_indicator_x_dest = DICE_1_INDICATOR_X;
	  } 
	  else if(OVER_DICE_2_BUTTON(event.motion.x-x,event.motion.y-y) && (dices[1]!=-1)){
	    dices_indicator_x_dest = DICE_2_INDICATOR_X;
	  } 
	  else if(OVER_DICE_3_BUTTON(event.motion.x-x,event.motion.y-y) && (dices[2]!=-1)){
	    dices_indicator_x_dest = DICE_3_INDICATOR_X;
	  } 
	  else if(can_reroll && OVER_REROLL_BUTTON(event.motion.x-x,event.motion.y-y)){
	    over_reroll = true;
	  }
	  break;
	}
      }
      break; // esac de SDL_MOUSEMOTION
    case SDL_MOUSEBUTTONDOWN:
      if( event.button.button==LEFT_BUTTON ){
		  if(OVER_TITLE_BAR(event.button.x,event.button.y)){
			  move_dialog = true;
			  update = true;}
	else { 
	  switch(type){
	  case DIALOG_TOSS:

	    switch(toss_state){
	    case TOSS_CHOICE:
	      if(OVER_HEAD_BUTTON(event.button.x-x,event.button.y-y) || OVER_HEAD_PIECE(event.motion.x-x,event.motion.y-y) ){
			face_choice = HEAD;
			toss_state = TOSS_UP;
			face_result = d2->roll();
			update = true;

			//SOUND - TALID
	   		#ifndef NOSOUND
			if(Config::is_sound_enabled() && sp){
				sp->play_sound(SND_COINFLIP, FX0_CHANNEL);
			}
			#endif
			//-----

	      } 
	      else if(OVER_TAIL_BUTTON(event.button.x-x,event.button.y-y) || OVER_TAIL_PIECE(event.motion.x-x,event.motion.y-y)){
			face_choice = TAIL;
			toss_state = TOSS_UP;
			face_result = d2->roll();
			update = true;
	   
			//SOUND - TALID
   			#ifndef NOSOUND
			if(Config::is_sound_enabled() && sp){
				sp->play_sound(SND_COINFLIP, FX0_CHANNEL);
			}
			#endif
			//-----
		  } 
	      break;
	    case TOSS_KICK_OR_RECEIVE:
	      if(OVER_KICK_BUTTON(event.button.x-x,event.button.y-y)){
			update = true;
			return (face_choice==face_result)?HOME_KICK:VISITOR_KICK;
	      } 
	      else if(OVER_RECEIVE_BUTTON(event.button.x-x,event.button.y-y)){
			update = true;
			return (face_choice==face_result)?VISITOR_KICK:HOME_KICK;
	      } 
	      break;
	    }
	    break;
	  case DIALOG_INFO:
	    if(OVER_NOT_AGREE(event.button.x-x,event.button.y-y) && can_complain){
	      update = true;
	      return NOT_AGREE;
	    } 
	    else if(OVER_CLOSE_BUTTON(event.button.x-x,event.button.y-y)){
	      update = true;
	      return CLOSE;
	    } 
	    break;
	  case DIALOG_YESNO:
	    if(OVER_YES_BUTTON(event.button.x-x,event.button.y-y)){
	      update = true;
	      return YES;
	    } 
	    else if(OVER_NO_BUTTON(event.button.x-x,event.button.y-y)){
	      update = true;
	      return NO;
	    } 
	    break;
	  case DIALOG_CHOICE:
	    if(OVER_A_BUTTON(event.button.x-x,event.button.y-y)){
	      update = true;
	      return CHOICE_A;
	    } 
	    else if(OVER_B_BUTTON(event.button.x-x,event.button.y-y)){
	      update = true;
	      return CHOICE_B;
	    } 
	    break;
	  case DIALOG_BLOCK:
	    if(OVER_DICE_1_BUTTON(event.button.x-x,event.button.y-y)  && (dices[0]!=-1)){
	      update = true;
	      return FIRST_DICE;
	    } 
	    else if(OVER_DICE_2_BUTTON(event.button.x-x,event.button.y-y) && (dices[1]!=-1)){
	      update = true;
	      return SECOND_DICE;
	    } 
	    else if(OVER_DICE_3_BUTTON(event.button.x-x,event.button.y-y) && (dices[2]!=-1) ){
	      update = true;
	      return THIRD_DICE;
	    } 
	    else if(OVER_REROLL_BUTTON(event.button.x-x,event.button.y-y) && can_reroll){
	      update = true;
	      return REROLL;
	    } 
	    break;
	  }
	}
      }
      break; // esac de SDL_MOUSEBUTTONDOWN
    case SDL_MOUSEBUTTONUP:
      update = true;
      move_dialog = false;
      break; // esac de SDL_MOUSEBUTTONUP
    }
  }
  
  return NOP;
}

void DialogBox::draw(){

  SDL_Rect r,s,d;
  static SDL_Rect old_r;
  r.x = x;
  r.y = y;
  r.w = DIALOG_W;
  r.h = DIALOG_H;

  if(move_dialog){
	cliprect.x = MAX(0,(x-DIALOG_W));
	cliprect.y = MAX(0,(y-DIALOG_H));
	cliprect.w = DIALOG_W*3;
	cliprect.h = DIALOG_H*3;
	SDL_SetClipRect(screen,&cliprect);
  }
  else if(update){
	cliprect.x = (Sint16) MAX(0,x-DIALOG_W/4);
	cliprect.y = (Sint16) MAX(0,y-DIALOG_H/4);
    cliprect.w = (Uint16) (DIALOG_W*1.5);
    cliprect.h = (Uint16) (DIALOG_H*1.5);
    SDL_SetClipRect(screen,&cliprect);
    update = false;
  }
  else {
    cliprect.x = 0;
    cliprect.y = 0;
    cliprect.w = 0;
    cliprect.h = 0;
    SDL_SetClipRect(screen,&cliprect);
    wait_for_next_frame();
    return;
  }

  // on commence par retablir l'arriere plan
  BlitSurface(old_screen,&old_r,screen,&old_r);

  // on draw le bg de la  boite
  BlitSurface(screen,&r,dialog_box_surf,NULL);
  BlitSurface(dialog_img[type],NULL,dialog_box_surf,NULL);

  switch(type){
  case DIALOG_TOSS:
    switch(toss_state){
    case TOSS_CHOICE:
      s.x = over_head?PIECES_W:0;
      s.y = 0;
      s.w = PIECES_W;
      s.h = PIECES_H;
      d.x = PIECE_HEAD_X;
      d.y = PIECE_HEAD_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(pieces_img,&s,dialog_box_surf,&d);
      
      if(over_head) {
		s.x = 0;
		s.y = 0;
		s.w = HEAD_BUTTON_W;
		s.h = HEAD_BUTTON_H;
		d.x = HEAD_BUTTON_X;
		d.y = HEAD_BUTTON_Y;
		d.w = s.w;
		d.h = s.h;
		BlitSurface(head_tail_img,&s,dialog_box_surf,&d);
      }
      
      s.x = over_tail?PIECES_W:0;
      s.y = PIECES_H;
      s.w = PIECES_W;
      s.h = PIECES_H;
      d.x = PIECE_TAIL_X;
      d.y = PIECE_TAIL_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(pieces_img,&s,dialog_box_surf,&d);
      
      if(over_tail) {
		s.x = 0;
		s.y = HEAD_BUTTON_H;
		s.w = TAIL_BUTTON_W;
		s.h = TAIL_BUTTON_H;
		d.x = TAIL_BUTTON_X;
		d.y = TAIL_BUTTON_Y;
		d.w = s.w;
		d.h = s.h;
		BlitSurface(head_tail_img,&s,dialog_box_surf,&d);
      }
      
      break;
    case TOSS_KICK_OR_RECEIVE:
      if(face_result==HEAD){
		 s.x = (face_choice==HEAD)?PIECES_W:0;
		 s.y = 0;
		 s.w = PIECES_W;
		 s.h = PIECES_H;
		 d.x = PIECE_HEAD_X;
		 d.y = PIECE_HEAD_Y;
		 d.w = s.w;
		 d.h = s.h;
		 BlitSurface(pieces_img,&s,dialog_box_surf,&d);
      }
      else {
		s.x = (face_choice==TAIL)?PIECES_W:0;
		s.y = PIECES_H;
		s.w = PIECES_W;
		s.h = PIECES_H;
		d.x = PIECE_TAIL_X;
		d.y = PIECE_TAIL_Y;
		d.w = s.w;
		d.h = s.h;
		BlitSurface(pieces_img,&s,dialog_box_surf,&d);
      }
      
      s.x = (face_choice==face_result)?(TOSS_BOTTOM_W):(2*TOSS_BOTTOM_W);
      s.y = 0;
      s.w = TOSS_BOTTOM_W;
      s.h = TOSS_BOTTOM_H;
      d.x = TOSS_BOTTOM_X;
      d.y = TOSS_BOTTOM_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(toss_bottom_img,&s,dialog_box_surf,&d);

      if(over_kick){
		 s.x = 0;
		 s.y = 0;
		 s.w = KICK_BUTTON_W;
		 s.h = KICK_BUTTON_H;
		 d.x = KICK_BUTTON_X;
		 d.y = KICK_BUTTON_Y;
		 d.w = s.w;
		 d.h = s.h;
		 BlitSurface(kick_receive_img,&s,dialog_box_surf,&d);
      }
      else if(over_receive){
		s.x = 0;
		s.y = KICK_BUTTON_H;
		s.w = RECEIVE_BUTTON_W;
		s.h = RECEIVE_BUTTON_H;
		d.x = RECEIVE_BUTTON_X;
		d.y = RECEIVE_BUTTON_Y;
		d.w = s.w;
		d.h = s.h;
		BlitSurface(kick_receive_img,&s,dialog_box_surf,&d);
      }

      break;
    case TOSS_UP:
      s.x = (face_choice==HEAD)?PIECES_W:0;
      s.y = 0;
      s.w = PIECES_W;
      s.h = PIECES_H;
      d.x = PIECE_HEAD_X;
      d.y = PIECE_HEAD_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(pieces_img,&s,dialog_box_surf,&d);

      s.x = (face_choice==TAIL)?PIECES_W:0;
      s.y = PIECES_H;
      s.w = PIECES_W;
      s.h = PIECES_H;
      d.x = PIECE_TAIL_X;
      d.y = PIECE_TAIL_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(pieces_img,&s,dialog_box_surf,&d);
      
      s.x = 0;
      s.y = 0;
      s.w = TOSS_BOTTOM_W;
      s.h = TOSS_BOTTOM_H;
      d.x = TOSS_BOTTOM_X;
      d.y = TOSS_BOTTOM_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(toss_bottom_img,&s,dialog_box_surf,&d);

      pieces_indicator_x = (pieces_indicator_x==PIECE_HEAD_INDICATOR_X)?PIECE_TAIL_INDICATOR_X:PIECE_HEAD_INDICATOR_X;

      update = true;
      
      if( pieces_toss_count++ >= 20 ){
		pieces_indicator_x = (face_result==HEAD)?PIECE_HEAD_INDICATOR_X:PIECE_TAIL_INDICATOR_X;
		toss_state = TOSS_KICK_OR_RECEIVE;
		title = strdup((face_choice==face_result)?team_name[HOME]:team_name[VISITOR]);
      }
      
      s.x = 0;
      s.y = 0;
      s.w = INDICATOR_W;
      s.h = INDICATOR_H;
      d.x = pieces_indicator_x;
      d.y = PIECES_INDICATOR_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(indicator_img,&s,dialog_box_surf,&d);
      break;
    }
    
    break;
  case DIALOG_INFO:
	  
	 if(can_complain){
      s.x = over_not_agree?NOT_AGREE_W:0;
      s.y = 0;
      s.w = NOT_AGREE_W;
      s.h = NOT_AGREE_H;
      d.x = NOT_AGREE_X;
      d.y = NOT_AGREE_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(not_agree_img,&s,dialog_box_surf,&d);
    }
    
    if(over_close) {
      s.x = 0;
      s.y = 0;
      s.w = CLOSE_BUTTON_W;
      s.h = CLOSE_BUTTON_H;
      d.x = CLOSE_BUTTON_X;
      d.y = CLOSE_BUTTON_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(close_img,&s,dialog_box_surf,&d);
    } 

    s.x = icons_id*ICONS_W;
    s.y = 0;
    s.w = ICONS_W;
    s.h = ICONS_H;
    d.x = ICONS_X;
    d.y = ICONS_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(icons_img,&s,dialog_box_surf,&d);
	
	draw_multiline_string(dialog_box_surf,MSG_X,MSG_Y,msg);

    break;
  case DIALOG_YESNO:

    if(over_yes) {
      s.x = 0;
      s.y = 0;
      s.w = YES_BUTTON_W;
      s.h = YES_BUTTON_H;
      d.x = YES_BUTTON_X;
      d.y = YES_BUTTON_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(yes_img,&s,dialog_box_surf,&d);
    }
    else if(over_no){
      s.x = 0;
      s.y = 0;
      s.w = NO_BUTTON_W;
      s.h = NO_BUTTON_H;
      d.x = NO_BUTTON_X;
      d.y = NO_BUTTON_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(no_img,&s,dialog_box_surf,&d);
    }

    s.x = icons_id*ICONS_W;
    s.y = 0;
    s.w = ICONS_W;
    s.h = ICONS_H;
    d.x = ICONS_X;
    d.y = ICONS_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(icons_img,&s,dialog_box_surf,&d);

	draw_multiline_string(dialog_box_surf,MSG_X,MSG_Y,msg);

    break;
  case DIALOG_CHOICE:

    if(over_a) {
      s.x = 0;
      s.y = 0;
      s.w = YES_BUTTON_W;
      s.h = YES_BUTTON_H;
      d.x = YES_BUTTON_X;
      d.y = YES_BUTTON_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(a_img,&s,dialog_box_surf,&d);
    }
    else if(over_b){
      s.x = 0;
      s.y = 0;
      s.w = NO_BUTTON_W;
      s.h = NO_BUTTON_H;
      d.x = NO_BUTTON_X;
      d.y = NO_BUTTON_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(b_img,&s,dialog_box_surf,&d);
    }
    s.x = icons_id*ICONS_W;
    s.y = 0;
    s.w = ICONS_W;
    s.h = ICONS_H;
    d.x = ICONS_X;
    d.y = ICONS_Y;
    d.w = s.w;
    d.h = s.h;
    BlitSurface(icons_img,&s,dialog_box_surf,&d);

	draw_multiline_string(dialog_box_surf,MSG_X,MSG_Y,msg);

    break;
  case DIALOG_BLOCK:

    if(dices_indicator_x_dest!=-1){
      s.x = 0;
      s.y = 0;
      s.w = INDICATOR_W;
      s.h = INDICATOR_H;
      if(dices_indicator_x_current < dices_indicator_x_dest ) {
		  update = true;
	dices_indicator_x_current+=DICES_INDICATOR_SPEED;
	if(dices_indicator_x_current >= dices_indicator_x_dest)
	  dices_indicator_x_current = dices_indicator_x_dest; 
      }
      else if(dices_indicator_x_current > dices_indicator_x_dest ) {
		update = true;
	dices_indicator_x_current-=DICES_INDICATOR_SPEED;
	if(dices_indicator_x_current <= dices_indicator_x_dest)
	  dices_indicator_x_current = dices_indicator_x_dest; 
      }
      d.x = dices_indicator_x_current;
      d.y = DICES_INDICATOR_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(indicator_img,&s,dialog_box_surf,&d);
    }
    
    if(who_choose==DEFENDER){
      s.x = 0;
      s.y = 0;
      s.w = DEFENDER_W;
      s.h = DEFENDER_H;
      d.x = DEFENDER_X;
      d.y = DEFENDER_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(defender_img,&s,dialog_box_surf,&d);
    }
    
    if(can_reroll){
      s.x = over_reroll?REROLL_BUTTON_W:0;
      s.y = 0;
      s.w = REROLL_BUTTON_W;
      s.h = REROLL_BUTTON_H;
      d.x = REROLL_BUTTON_X;
      d.y = REROLL_BUTTON_Y;
      d.w = s.w;
      d.h = s.h;
      BlitSurface(reroll_img,&s,dialog_box_surf,&d);
    }
    
    int i=0;

    while(i<3){
      if(dices[i]!=-1){
		s.x = dices[i]*DICES_BUTTON_W;
		s.y = 0;
		s.w = DICES_BUTTON_W;
		s.h = DICES_BUTTON_H;
		d.x = DICE_1_BUTTON_X+(i*DICES_DIFF);
		d.y = DICE_1_BUTTON_Y;
		d.w = s.w;
		d.h = s.h;
		BlitSurface(block_dices_img,&s,dialog_box_surf,&d);
      }
      i++;
    }
    
    break;
  }

  draw_title(dialog_box_surf);

  // et on flippe
  SDL_SetAlpha(dialog_box_surf,ALPHA_FLAGS,(move_dialog)?(DIALOG_ALPHA/2):DIALOG_ALPHA);
  BlitSurface(dialog_box_surf,NULL,screen,&r);
  
  SDL_Flip(screen);
  wait_for_next_frame(); 

  old_r = r;

} 
   
void DialogBox::wait_for_next_frame(){
  static Uint32 next_tick = 0;
  Uint32 this_tick;
  
  /* Wait for the next frame */
  this_tick = SDL_GetTicks(); 
  if ( this_tick < next_tick ) {
    SDL_Delay(next_tick-this_tick);
  }
  next_tick = this_tick + (1000/DIALOG_FPS);
}

SDL_Surface* DialogBox::load_image(const char *datafile, int transparent){
  SDL_Surface *image;
  image = IMG_Load(datafile);
  if ( image == NULL ) {
    fprintf(stderr, "Couldn't load image %s: %s\n",datafile, IMG_GetError());
    return NULL;
  }
   
  if ( transparent ) {
    /* Assuming 8-bit BMP image */
    SDL_SetColorKey(image, (SDL_SRCCOLORKEY|SDL_RLEACCEL),*(Uint8 *)image->pixels);
  }
  
  SDL_Surface *r =  SDL_DisplayFormat(image);
  SDL_FreeSurface(image);
  return r;
}

void DialogBox::draw_multiline_string(SDL_Surface* dialog_box_surf,int msg_x,int msg_y,const char *msg){
	
	int l=0;
	unsigned int debut=0;
	unsigned int i=0,j=0;
	unsigned int espace=0;
	char buffer[MSG_MAX_CHAR];
	memset(buffer,0,MSG_MAX_CHAR);
	bool eos = false,saut_ligne = false;;
	while(!eos){
		saut_ligne = false;
		memset(buffer,0,MSG_MAX_CHAR);
		// trim gauche de la chaine
		while ( msg[debut] && msg[debut]==' ' ) {
			debut++; 
			espace = debut;
		}

		// recherche espace ou fin de chaine
		for (i=debut;i<debut+MSG_MAX_CHAR;i++){
			if(!msg[i]) {eos=true;break;}
			if(msg[i]=='\\' && msg[i+1] && msg[i+1]=='n'){
				espace = i;
				saut_ligne = true;
				break;
			}
			if(msg[i]==' ') espace=i;
		}
		
		// all right espace<fin chaine, print(debut,espace), ligne++
		if(!eos) {
			// espace != debut
			if( espace!=debut ){
				
				for(j=debut;j<espace;j++) 
					buffer[j-debut] = msg[j];
				buffer[espace-debut+1] ='\0';

				PutString(dialog_box_surf,MSG_X,l*MSG_INTERLIGNE + MSG_Y,buffer);		

				debut = espace + (saut_ligne?2:0);
				l++;
			} 
			else { // == cad mot trop long
				
				for(j=debut;j<debut+MSG_MAX_CHAR-2;j++) 
					buffer[j-debut] = msg[j];
				buffer[MSG_MAX_CHAR-2] ='-';
				buffer[MSG_MAX_CHAR-1] ='\0';

				PutString(dialog_box_surf,MSG_X,l*MSG_INTERLIGNE + MSG_Y,buffer);		

				debut += MSG_MAX_CHAR-2;
				l++;		
			}
		}
		else {
			for(j=debut;j<strlen(msg);j++) 
					buffer[j-debut] = msg[j];
			buffer[strlen(msg)-debut] ='\0';

			PutString(dialog_box_surf,MSG_X,l*MSG_INTERLIGNE + MSG_Y,buffer);		
		
		}
		// fin du message print(debut,fin)
	}
}

void DialogBox::draw_title(SDL_Surface *s){

	if(strlen(title) > TITLE_MAX_CHAR ){
		title[TITLE_MAX_CHAR-3] = '.';
		title[TITLE_MAX_CHAR-2] = '.';
		title[TITLE_MAX_CHAR-1] = '.';
		title[TITLE_MAX_CHAR] = '\0';
	}
	XCenteredString(s, TITLE_Y, title);
}

void DialogBox::change_language(const char* lg){
	translator->set_language(lg);
}

#ifndef NOSOUND
void DialogBox::set_sound_player(SoundPlayer* sp_m){
	if(Config::is_sound_enabled()){
		this->sp = sp_m;
	}
}
#endif

