/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.autosaveandbackup;

import com.google.common.eventbus.Subscribe;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.jabref.logic.bibtex.InvalidFieldValueException;
import org.jabref.logic.exporter.BibtexDatabaseWriter;
import org.jabref.logic.exporter.FileSaveSession;
import org.jabref.logic.exporter.SaveException;
import org.jabref.logic.exporter.SavePreferences;
import org.jabref.logic.util.io.FileUtil;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.database.event.BibDatabaseContextChangedEvent;
import org.jabref.model.database.event.CoarseChangeFilter;
import org.jabref.preferences.JabRefPreferences;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackupManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackupManager.class);
    private static final String BACKUP_EXTENSION = ".sav";
    private static Set<BackupManager> runningInstances = new HashSet<BackupManager>();
    private final BibDatabaseContext bibDatabaseContext;
    private final JabRefPreferences preferences;
    private final ExecutorService executor;
    private final Runnable backupTask = () -> this.determineBackupPath().ifPresent(this::performBackup);
    private final CoarseChangeFilter changeFilter;

    private BackupManager(BibDatabaseContext bibDatabaseContext) {
        this.bibDatabaseContext = bibDatabaseContext;
        this.preferences = JabRefPreferences.getInstance();
        ArrayBlockingQueue<Runnable> workerQueue = new ArrayBlockingQueue<Runnable>(1);
        this.executor = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.SECONDS, workerQueue);
        this.changeFilter = new CoarseChangeFilter(bibDatabaseContext);
        this.changeFilter.registerListener(this);
    }

    static Path getBackupPath(Path originalPath) {
        return FileUtil.addExtension(originalPath, BACKUP_EXTENSION);
    }

    public static BackupManager start(BibDatabaseContext bibDatabaseContext) {
        BackupManager backupManager = new BackupManager(bibDatabaseContext);
        backupManager.startBackupTask();
        runningInstances.add(backupManager);
        return backupManager;
    }

    public static void shutdown(BibDatabaseContext bibDatabaseContext) {
        runningInstances.stream().filter(instance -> instance.bibDatabaseContext == bibDatabaseContext).forEach(BackupManager::shutdown);
        runningInstances.removeIf(instance -> instance.bibDatabaseContext == bibDatabaseContext);
    }

    public static boolean checkForBackupFile(Path originalPath) {
        Path backupPath = BackupManager.getBackupPath(originalPath);
        return Files.exists(backupPath, new LinkOption[0]) && !Files.isDirectory(backupPath, new LinkOption[0]);
    }

    public static void restoreBackup(Path originalPath) {
        Path backupPath = BackupManager.getBackupPath(originalPath);
        try {
            Files.copy(backupPath, originalPath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            LOGGER.error("Error while restoring the backup file.", e);
        }
    }

    private Optional<Path> determineBackupPath() {
        return this.bibDatabaseContext.getDatabasePath().map(BackupManager::getBackupPath);
    }

    private void performBackup(Path backupPath) {
        try {
            Charset charset = this.bibDatabaseContext.getMetaData().getEncoding().orElse(this.preferences.getDefaultEncoding());
            SavePreferences savePreferences = this.preferences.loadForSaveFromPreferences().withEncoding(charset).withMakeBackup(false);
            ((FileSaveSession)new BibtexDatabaseWriter<FileSaveSession>(FileSaveSession::new).saveDatabase(this.bibDatabaseContext, savePreferences)).commit(backupPath);
        }
        catch (SaveException e) {
            this.logIfCritical(e);
        }
    }

    private void logIfCritical(SaveException e) {
        Throwable innermostCause = e;
        while (innermostCause.getCause() != null) {
            innermostCause = innermostCause.getCause();
        }
        boolean isErrorInField = innermostCause instanceof InvalidFieldValueException;
        if (!isErrorInField) {
            LOGGER.error("Error while saving file.", e);
        }
    }

    @Subscribe
    public synchronized void listen(BibDatabaseContextChangedEvent event) {
        this.startBackupTask();
    }

    private void startBackupTask() {
        try {
            this.executor.submit(this.backupTask);
        }
        catch (RejectedExecutionException e) {
            LOGGER.debug("Rejecting while another backup process is already running.");
        }
    }

    private void shutdown() {
        this.changeFilter.unregisterListener(this);
        this.changeFilter.shutdown();
        this.executor.shutdown();
        this.determineBackupPath().ifPresent(this::deleteBackupFile);
    }

    private void deleteBackupFile(Path backupPath) {
        try {
            if (Files.exists(backupPath, new LinkOption[0]) && !Files.isDirectory(backupPath, new LinkOption[0])) {
                Files.delete(backupPath);
            }
        }
        catch (IOException e) {
            LOGGER.error("Error while deleting the backup file.", e);
        }
    }
}

