/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.shared;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import javafx.collections.ObservableList;
import org.jabref.logic.exporter.BibDatabaseWriter;
import org.jabref.logic.exporter.MetaDataSerializer;
import org.jabref.logic.importer.ParseException;
import org.jabref.logic.importer.util.MetaDataParser;
import org.jabref.logic.shared.DBMSProcessor;
import org.jabref.logic.shared.event.ConnectionLostEvent;
import org.jabref.logic.shared.event.SharedEntryNotPresentEvent;
import org.jabref.logic.shared.event.UpdateRefusedEvent;
import org.jabref.logic.shared.exception.OfflineLockException;
import org.jabref.model.bibtexkeypattern.GlobalBibtexKeyPattern;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.database.event.EntryAddedEvent;
import org.jabref.model.database.event.EntryRemovedEvent;
import org.jabref.model.database.shared.DatabaseConnection;
import org.jabref.model.database.shared.DatabaseConnectionProperties;
import org.jabref.model.database.shared.DatabaseNotSupportedException;
import org.jabref.model.database.shared.DatabaseSynchronizer;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.event.EntryEvent;
import org.jabref.model.entry.event.EntryEventSource;
import org.jabref.model.entry.event.FieldChangedEvent;
import org.jabref.model.metadata.MetaData;
import org.jabref.model.metadata.event.MetaDataChangedEvent;
import org.jabref.model.util.FileUpdateMonitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DBMSSynchronizer
implements DatabaseSynchronizer {
    private static final Logger LOGGER = LoggerFactory.getLogger(DBMSSynchronizer.class);
    private DBMSProcessor dbmsProcessor;
    private String dbName;
    private final BibDatabaseContext bibDatabaseContext;
    private MetaData metaData;
    private final BibDatabase bibDatabase;
    private final EventBus eventBus;
    private Connection currentConnection;
    private final Character keywordSeparator;
    private final GlobalBibtexKeyPattern globalCiteKeyPattern;
    private FileUpdateMonitor fileMonitor;

    public DBMSSynchronizer(BibDatabaseContext bibDatabaseContext, Character keywordSeparator, GlobalBibtexKeyPattern globalCiteKeyPattern, FileUpdateMonitor fileMonitor) {
        this.bibDatabaseContext = Objects.requireNonNull(bibDatabaseContext);
        this.bibDatabase = bibDatabaseContext.getDatabase();
        this.metaData = bibDatabaseContext.getMetaData();
        this.fileMonitor = fileMonitor;
        this.eventBus = new EventBus();
        this.keywordSeparator = keywordSeparator;
        this.globalCiteKeyPattern = Objects.requireNonNull(globalCiteKeyPattern);
    }

    @Subscribe
    public void listen(EntryAddedEvent event) {
        if (this.isEventSourceAccepted(event) && this.checkCurrentConnection()) {
            this.synchronizeLocalMetaData();
            this.synchronizeLocalDatabase();
            this.dbmsProcessor.insertEntry(event.getBibEntry());
        }
    }

    @Subscribe
    public void listen(FieldChangedEvent event) {
        if (this.isPresentLocalBibEntry(event.getBibEntry()) && this.isEventSourceAccepted(event) && this.checkCurrentConnection()) {
            this.synchronizeLocalMetaData();
            BibEntry bibEntry = event.getBibEntry();
            this.synchronizeSharedEntry(bibEntry);
            this.synchronizeLocalDatabase();
        }
    }

    @Subscribe
    public void listen(EntryRemovedEvent event) {
        if (this.isEventSourceAccepted(event) && this.checkCurrentConnection()) {
            this.dbmsProcessor.removeEntry(event.getBibEntry());
            this.synchronizeLocalMetaData();
            this.synchronizeLocalDatabase();
        }
    }

    @Subscribe
    public void listen(MetaDataChangedEvent event) {
        if (this.checkCurrentConnection()) {
            this.synchronizeSharedMetaData(event.getMetaData(), this.globalCiteKeyPattern);
            this.synchronizeLocalDatabase();
            this.applyMetaData();
            this.dbmsProcessor.notifyClients();
        }
    }

    @Subscribe
    public void listen(EntryEvent event) {
        if (this.isEventSourceAccepted(event)) {
            this.dbmsProcessor.notifyClients();
        }
    }

    public void initializeDatabases() throws DatabaseNotSupportedException {
        try {
            if (!this.dbmsProcessor.checkBaseIntegrity()) {
                LOGGER.info("Integrity check failed. Fixing...");
                this.dbmsProcessor.setupSharedDatabase();
                if (this.dbmsProcessor.checkForPre3Dot6Intergrity()) {
                    throw new DatabaseNotSupportedException();
                }
            }
        }
        catch (SQLException e) {
            throw new IllegalStateException(e);
        }
        this.dbmsProcessor.startNotificationListener(this);
        this.synchronizeLocalMetaData();
        this.synchronizeLocalDatabase();
    }

    @Override
    public void synchronizeLocalDatabase() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        ObservableList<BibEntry> localEntries = this.bibDatabase.getEntries();
        Map<Integer, Integer> idVersionMap = this.dbmsProcessor.getSharedIDVersionMapping();
        this.removeNotSharedEntries((List<BibEntry>)localEntries, idVersionMap.keySet());
        for (Map.Entry<Integer, Integer> idVersionEntry : idVersionMap.entrySet()) {
            Optional<BibEntry> bibEntry;
            boolean match = false;
            for (BibEntry localEntry : localEntries) {
                Optional<BibEntry> sharedEntry;
                if (idVersionEntry.getKey().intValue() != localEntry.getSharedBibEntryData().getSharedID()) continue;
                match = true;
                if (idVersionEntry.getValue() <= localEntry.getSharedBibEntryData().getVersion() || !(sharedEntry = this.dbmsProcessor.getSharedEntry(idVersionEntry.getKey())).isPresent()) continue;
                localEntry.setType(sharedEntry.get().getType(), EntryEventSource.SHARED);
                localEntry.getSharedBibEntryData().setVersion(sharedEntry.get().getSharedBibEntryData().getVersion());
                for (String field2 : sharedEntry.get().getFieldNames()) {
                    localEntry.setField(field2, sharedEntry.get().getField(field2), EntryEventSource.SHARED);
                }
                Set<String> redundantLocalEntryFields = localEntry.getFieldNames();
                redundantLocalEntryFields.removeAll(sharedEntry.get().getFieldNames());
                for (String redundantField : redundantLocalEntryFields) {
                    localEntry.clearField(redundantField, EntryEventSource.SHARED);
                }
            }
            if (match || !(bibEntry = this.dbmsProcessor.getSharedEntry(idVersionEntry.getKey())).isPresent()) continue;
            this.bibDatabase.insertEntry(bibEntry.get(), EntryEventSource.SHARED);
        }
    }

    private void removeNotSharedEntries(List<BibEntry> localEntries, Set<Integer> sharedIDs) {
        for (int i = 0; i < localEntries.size(); ++i) {
            BibEntry localEntry = localEntries.get(i);
            boolean match = false;
            for (int sharedID : sharedIDs) {
                if (localEntry.getSharedBibEntryData().getSharedID() != sharedID) continue;
                match = true;
                break;
            }
            if (match) continue;
            this.eventBus.post(new SharedEntryNotPresentEvent(localEntry));
            this.bibDatabase.removeEntry(localEntry, EntryEventSource.SHARED);
            --i;
        }
    }

    @Override
    public void synchronizeSharedEntry(BibEntry bibEntry) {
        if (!this.checkCurrentConnection()) {
            return;
        }
        try {
            BibDatabaseWriter.applySaveActions(bibEntry, this.metaData);
            this.dbmsProcessor.updateEntry(bibEntry);
        }
        catch (OfflineLockException exception) {
            this.eventBus.post(new UpdateRefusedEvent(this.bibDatabaseContext, exception.getLocalBibEntry(), exception.getSharedBibEntry()));
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error: ", e);
        }
    }

    public void synchronizeLocalMetaData() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        try {
            this.metaData.setEventPropagation(false);
            MetaDataParser parser = new MetaDataParser(this.fileMonitor);
            parser.parse(this.metaData, this.dbmsProcessor.getSharedMetaData(), this.keywordSeparator);
            this.metaData.setEventPropagation(true);
        }
        catch (ParseException e) {
            LOGGER.error("Parse error", e);
        }
    }

    private void synchronizeSharedMetaData(MetaData data, GlobalBibtexKeyPattern globalCiteKeyPattern) {
        if (!this.checkCurrentConnection()) {
            return;
        }
        try {
            this.dbmsProcessor.setSharedMetaData(MetaDataSerializer.getSerializedStringMap(data, globalCiteKeyPattern));
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error: ", e);
        }
    }

    public void applyMetaData() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        for (BibEntry bibEntry : this.bibDatabase.getEntries()) {
            if (BibDatabaseWriter.applySaveActions(bibEntry, this.metaData).isEmpty()) continue;
            try {
                this.dbmsProcessor.updateEntry(bibEntry);
            }
            catch (OfflineLockException exception) {
                this.eventBus.post(new UpdateRefusedEvent(this.bibDatabaseContext, exception.getLocalBibEntry(), exception.getSharedBibEntry()));
            }
            catch (SQLException e) {
                LOGGER.error("SQL Error: ", e);
            }
        }
    }

    @Override
    public void pullChanges() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        this.synchronizeLocalDatabase();
        this.synchronizeLocalMetaData();
    }

    public boolean checkCurrentConnection() {
        try {
            boolean isValid = this.currentConnection.isValid(0);
            if (!isValid) {
                this.eventBus.post(new ConnectionLostEvent(this.bibDatabaseContext));
            }
            return isValid;
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error:", e);
            return false;
        }
    }

    public boolean isEventSourceAccepted(EntryEvent event) {
        EntryEventSource eventSource = event.getEntryEventSource();
        return eventSource == EntryEventSource.LOCAL || eventSource == EntryEventSource.UNDO;
    }

    @Override
    public void openSharedDatabase(DatabaseConnection connection) throws DatabaseNotSupportedException {
        this.dbName = connection.getProperties().getDatabase();
        this.currentConnection = connection.getConnection();
        this.dbmsProcessor = DBMSProcessor.getProcessorInstance(connection);
        this.initializeDatabases();
    }

    @Override
    public void closeSharedDatabase() {
        try {
            this.dbmsProcessor.stopNotificationListener();
            this.currentConnection.close();
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error:", e);
        }
    }

    private boolean isPresentLocalBibEntry(BibEntry bibEntry) {
        return this.bibDatabase.getEntries().contains((Object)bibEntry);
    }

    @Override
    public String getDBName() {
        return this.dbName;
    }

    public DBMSProcessor getDBProcessor() {
        return this.dbmsProcessor;
    }

    @Override
    public DatabaseConnectionProperties getConnectionProperties() {
        return this.dbmsProcessor.getDBMSConnectionProperties();
    }

    public void setMetaData(MetaData metaData) {
        this.metaData = metaData;
    }

    @Override
    public void registerListener(Object listener) {
        this.eventBus.register(listener);
    }
}

