package Agent;

#
# Agent.pm module implements Agent interface
#

use strict;
use warnings;

use Logging;
use XmlNode;
use AgentConfig;
use ContentDumperBase;
use ContentDumper;
use DatabaseContentDumper;
use Db::Connection;
use Db::DbConnect;
use Db::MysqlUtils;
use Dumper;
use DumperUtils;
use Storage::Storage;
use EncodeBase64;
use SpamAssassinCfg;
use CommonXmlNodes;
use PreMigration;
use PreMigrationChecks;

my $agentWorkDir;

use constant FTP_USERNAME_LENGTH_LIMIT => 16;     # this is Plesk limitation
use constant FTP_USERNAME_INDEX_LENGTH => 4;

my $do_gzip = undef;
sub enableCompression {
  $do_gzip = 1;
}

sub getCompressionStatus {
  return $do_gzip;
}

#
# Begin interface methods
#
sub setWorkDir {
  my $workDir = shift;
  $agentWorkDir = $workDir;
}

sub getWorkDir {
  return $agentWorkDir || AgentConfig::cwd();;
}

my $contentDumperBase = ContentDumperBase->new(Storage::Storage::createFileStorage($do_gzip, getWorkDir()));
sub getContentTransportDescription{
  return $contentDumperBase->getContentTransportDescription();
}

sub getAgentName {
  return "PPCPL";
}

sub getAgentVersion {
  return '10.4';
}

sub getAgentStatus {
  Logging::trace("Getting status...");

  return Dumper::getAgentStatus();
}

sub getAdminPreferences {
  return undef;
}

# Should return an array of reseller identifiers, that could be a number, name or guid.
# Should be unique through migration dump.
sub getResellers {
  Logging::trace("Getting resellers...");

  return Dumper::getResellers();
}

sub getReseller {
  # reseller identifier from 'getResellers' result
  my $resellerId = shift;

  Logging::trace("Getting reseller dump for '$resellerId' reseller...");

  my $generalInfo = Dumper::getResellerGeneralInfo($resellerId);

  my %limits = _getResellerLimits($resellerId);
  my %permissions = _getResellerPermissions($resellerId);

  return XmlNode->new('reseller', (
    'attributes' => {
      'id' => $resellerId,
      'name' => $resellerId,
      'contact' => $generalInfo->{'fullname'}, 
      'guid' => ''
    },
    'children' => [
      _makePreferencesNode($generalInfo->{'email'}),
      XmlNode->new('properties', (
        'children' => [
          CommonXmlNodes::encryptedPassword($generalInfo->{'password'}),
          CommonXmlNodes::status($generalInfo->{'enabled'} eq '1', 'admin')
        ]
      )),
      CommonXmlNodes::limitsAndPermissions(\%limits, \%permissions),
      CommonXmlNodes::ipPool(getResellerIPPool($resellerId))
    ]
  ));
}

# Should return an array of clients identifiers, that could be a number, name or guid.
# Should be unique through migration dump
sub getClients {
  # reseller that owns the clients returned.
  # Could be 'undef' for default reseller ('root' or 'admin')
  my $owner = shift;

  Logging::trace(
    "Getting clients list for '" . ( defined($owner) ? $owner : 'undef' ) . "' reseller ... " );

  return Dumper::getClientIdsForReseller($owner);
}

# @param clientId - consists of domain name alone
sub getClient {
  my ( $clientId, $dumpOptions ) = @_;

  # client identifier from 'getClients' result,
  Logging::trace("Getting client dump for '$clientId' client...");
  
  my $domain = $clientId;
  my $siteId = Dumper::getSiteIdByDomain($domain);
  my $usersInfoMapPtr = Dumper::getSiteUsersInfoMap($domain);
  my $clientName = Dumper::getSiteAdminLoginByDomain($domain);

  PreMigrationChecks::checkClient($domain);

  return XmlNode->new('client',
    'attributes' => {
      'id' => $clientName, 
      'name' => $clientName, 
      'guid' => ''
    },
    'children' => [ 
      _makePreferencesNode(
        Dumper::getSiteContactEmailByDomain($domain)
      ),
      XmlNode->new('properties',
        'children' => [
          CommonXmlNodes::encryptedPassword(
            $usersInfoMapPtr->{$clientName}{'password'}
          ),
          CommonXmlNodes::status(!Dumper::isDisabledSite($siteId), 'admin')
        ]
      ),
      CommonXmlNodes::ipPool(getClientIPPool($clientId)),
      XmlNode->new('domains', 
        'children' => [ 
          _getDomainNode($domain)
        ]
      )
    ]
  );
}

# Only whole client migration is supported, so:
# - returning empty list of domains to make domains selection impossible
# - domain will be attached to appropriate client node during full dump
sub getDomains {
  return ();
}

sub getPreMigrationDomains {
  my ($owner) = @_;

  # attach domains to a reseller or an admin user
  # so pre-migration tree is more close to the source PPCPL system 
  # where there is no separate "client" entity
  if (!defined($owner) || scalar(grep { $_ eq $owner } Dumper::getResellers()) > 0) {
    # in PPCPL migration client == domain, so get list of clients instead of list of domains
    return Dumper::getClientIdsForReseller($owner);
  }
}

sub getServerNode {
  my @children;
  my $maxSpamassassinThreads = Dumper::getSpamassassinMaxThreads();
  if (defined($maxSpamassassinThreads)) {
    push @children,
      XmlNode->new('spamassassin',
        'attributes' => {
          'max-spam-threads' => $maxSpamassassinThreads
        }
      );
  }

  if (@children) {
    return XmlNode->new('server', 'children' => \@children);
  } else {
    return undef;
  }
}

sub getResellerIPPool {
  my ($resellerName) = @_;

  my @clientNames = getClients($resellerName);
  return [ map { @{getClientIPPool($_)} } @clientNames ];
}

# @param clientId - consists of domain name alone
sub getClientIPPool {
  my ($clientId) = @_;

  return [ _getSiteIPInfo(Dumper::getSiteIdByDomain(my $domain = $clientId)) ];
}

#
# End of interface methods
#



sub _getDomainNode {
  my ($domainId) = @_;

  Logging::trace("Getting dump of '$domainId' domain");

  my $domainNode = XmlNode->new('domain');
  $domainNode->setAttribute( 'name', $domainId );
  $domainNode->setAttribute( 'guid', '' );

  $domainNode->addChild( XmlNode->new('preferences') );
  my $domainPropertiesNode = _getDomainPropertiesNode($domainId);
  $domainNode->addChild($domainPropertiesNode);

  my %limits = _getDomainLimits($domainId);
  my %permissions = _getDomainPermissions($domainId);
  $domainNode->addChild(
    CommonXmlNodes::limitsAndPermissions(\%limits, \%permissions)
  );

  my $siteUsersInfo = Dumper::getSiteUsersInfoMap($domainId);

  $domainNode->addChild(_getMailSystem($domainId, $siteUsersInfo));
  $domainNode->addChild(_getDatabases($domainId));
  $domainNode->addChild(_getMaillists($domainId));
  $domainNode->addChild(_getSslCertificates($domainId));
  $domainNode->addChild(_getDomainPHosting($domainId, $siteUsersInfo));

  return $domainNode;
}

sub _getDomainPropertiesNode {
  my ($domain) = @_;

  Logging::trace("Getting properties for '$domain' domain ... ");

  my $domainId             = Dumper::getSiteIdByDomain($domain);
  my $domainPropertiesNode = XmlNode->new('properties');

  my $ipInfo = _getSiteIPInfo($domainId);
  if (defined $ipInfo) {
    $domainPropertiesNode->addChild(CommonXmlNodes::ip($ipInfo->{'ip'}, $ipInfo->{'type'}));
  }

  my $isEnabled = !Dumper::isDisabledSite($domainId);

  PreMigration::assert(!defined $ipInfo && $isEnabled, 'ENABLED_SITE_HAS_NO_IPINFO', {'domain' => $domain});

  $domainPropertiesNode->addChild(CommonXmlNodes::status($isEnabled, 'admin'));
  $domainPropertiesNode->addChild(_makeDnsZoneNode($domainId));

  return $domainPropertiesNode;
}

sub _getDomainPHosting {
  my ($domainName, $usersInfoMapPtr) = @_;

  Logging::trace("Getting phosting for domain '$domainName' ... ");
  my $siteId = Dumper::getSiteIdByDomain($domainName);
  my %ipinfo = Dumper::getSpecificInfoForSite($siteId, 'ipinfo');
  my %ssi = Dumper::getSpecificInfoForSite($siteId, 'ssi');
  my %cgi = Dumper::getSpecificInfoForSite($siteId, 'cgi');
  my %perl = Dumper::getSpecificInfoForSite($siteId, 'mod_perl');
  my %mm4 = Dumper::getSpecificInfoForSite($siteId, 'mivamerchant');
  my %mm5 = Dumper::getSpecificInfoForSite($siteId, 'mivamerchant5');
  my %frontpage = Dumper::getSpecificInfoForSite($siteId, 'frontpage');

  my $pHostingNode = XmlNode->new('phosting',
    'attributes' => {
      'guid' =>           '',
      'owner-guid' =>     '',
      'wu_script' =>      'true',
      'https' =>          $ipinfo{'namebased'} ? 'false' : 'true',
      'fp' =>             $frontpage{'enabled'} ? 'true' : 'false',
      'fpssl' =>          $frontpage{'enabled'} ? 'true' : 'false',
      'fpauth' =>         $frontpage{'enabled'} ? 'true' : 'false',
      'webstat' =>        'true',
      'errdocs' =>        'false',
      'shared-content' => 'true',
      'www-root' =>	  'httpdocs'
    }
  );

  $pHostingNode->addChild(ContentDumper::getPHostingContent($contentDumperBase, $domainName));

  $pHostingNode->addChild(_makeHostingPreferencesNode($domainName, $usersInfoMapPtr));

  $pHostingNode->addChild(
    XmlNode->new('limits-and-permissions',
      'children' => [
        _makePHostingScriptingNode(\%ssi, \%cgi, \%perl, \%mm4, \%mm5, \%frontpage)
      ]
    )
  );

  $pHostingNode->addChild(
    XmlNode->new('webusers',
      'children' => _getWebUsers($domainName, \%ssi, \%cgi, \%perl, \%frontpage)
    )
  );

  my %siteInfo = Dumper::getSpecificInfoForSite($siteId, 'siteinfo');
  my ($subdomainsArray, $ftpusersArray) = _makeSubdomainsAndFtpusersNodes($domainName,
    $pHostingNode, $usersInfoMapPtr, \%siteInfo, \%ipinfo, \%ssi, \%perl, \%mm4, \%mm5);

  if (defined($ftpusersArray)) {
    $pHostingNode->addChild(XmlNode->new('ftpusers',
        'children' => [ @{$ftpusersArray} ]
    ));
  }

  if (defined($subdomainsArray)) {
    $pHostingNode->addChild(XmlNode->new('subdomains',
        'children' => [ @{$subdomainsArray} ]
    ));
  }

  return $pHostingNode;
}

sub _makeSubdomainsAndFtpusersNodes {
  my ($domainName, $pHostingNode, $usersInfoMapPtr, $siteInfoPtr, $ipinfoPtr, $ssi, $perl, $mm4, $mm5) = @_;

  my $listOfSubdomains = Dumper::getListOfSubdomains($domainName) || return undef;
  my $siteAdminLogin = $siteInfoPtr->{'admin_user'};
  my %usedUsernames = map { $_ => 1 } keys %{$listOfSubdomains};
  my (@subdomains, @ftpusers);

  while (my ($subdomainName, $subdomainInfo) = each %{$listOfSubdomains}) {
    my $owner = $subdomainInfo->{owner};
    my $subdomainDir = "subdomains_wwwroot/$subdomainName";

    $owner = $siteAdminLogin if (!defined($owner) || $owner eq "");

    if ($owner ne $siteAdminLogin) {
        my $ftpusername = _generateFtpuserName("${owner}_$subdomainName", \%usedUsernames);

        unless (defined($ftpusername)) {
            PreMigration::message('CANNOT_GENERATE_FTPUSER_NAME', {'domain' => $subdomainName,
                    'owner' => $owner, 'maxLength' => FTP_USERNAME_LENGTH_LIMIT});
            next;
        }

        push @ftpusers, XmlNode->new('ftpuser',
            'attributes' => {
                'name' => $ftpusername
            },
            'children' => [
                CommonXmlNodes::sysuser($ftpusername, $usersInfoMapPtr->{$owner}{'password'}, "/$subdomainDir"),
            ]
        );
    }

    push @subdomains, XmlNode->new('subdomain', (
        'attributes' => {
                'name'             => $subdomainName,
                'id'               => '',
                'guid'             => '',
                'owner-guid'       => '',
                'shared-content'   => "true",
                'https'            => $ipinfoPtr->{'namebased'} ? 'false' : 'true',     # get value from parent domain
                'www-root'         => "$subdomainDir",
                    ###
                    # PPCPL does not support frontpage for subdomains
                'fp'               => "false",
                'fpssl'            => "false",
                'fpauth'           => "false"
        },
        'children' => [
                ContentDumper::getSubdomainPHostingContent($contentDumperBase, $domainName, $pHostingNode,
                            $subdomainName, $subdomainInfo->{document_root},
                            $subdomainInfo->{cgi_root}),
                _makeSubdomainScriptingNode($subdomainInfo, $ssi, $perl, $mm4, $mm5)
        ]
    ));
  }

  return (\@subdomains, \@ftpusers);
}

sub _generateFtpuserName {
  my ($ftpusername, $usedUsernamesPtr) = @_;
  my $index = 0;

  if (length($ftpusername) > FTP_USERNAME_LENGTH_LIMIT) {
    $ftpusername = _getStringIndexWithSuffix($ftpusername, ++$index, FTP_USERNAME_LENGTH_LIMIT, FTP_USERNAME_INDEX_LENGTH);
  }
  return undef if (!defined($ftpusername));

  # check collisions with other ftpusernames and subdomains (if already presented)
  while ($usedUsernamesPtr->{$ftpusername}) {
    $ftpusername = _getStringIndexWithSuffix($ftpusername, ++$index, FTP_USERNAME_LENGTH_LIMIT, FTP_USERNAME_INDEX_LENGTH);
  }

  if (defined($ftpusername)) {
      $usedUsernamesPtr->{$ftpusername} = 1;
  }

  return $ftpusername;
}

sub _getStringIndexWithSuffix {
  my ($string, $valueIndex, $maxLength, $indexStrLength) = @_;

  return undef if ($valueIndex >= 10 ** $indexStrLength);

  my $strIndex = sprintf("%0${indexStrLength}d", $valueIndex);

  return (substr($string, 0, $maxLength - $indexStrLength) . $strIndex);
}

sub _getSslCertificates {
  my ($domainName) = @_;

  my $siteId = Dumper::getSiteIdByDomain($domainName);
  my %openssl = Dumper::getSpecificInfoForSite($siteId, 'openssl');
  unless ($openssl{'enabled'}) {
    return;
  }

  my $httpdConfDir = Dumper::getFilesystemByDomainName($domainName) . "/etc/httpd/conf";
  my @certInfoMap = (
    { 'name' => 'certificate-data', 'file' => $httpdConfDir . "/ssl.crt/server.crt"},
    { 'name' => 'signing-request', 'file' => $httpdConfDir . "/ssl.csr/server.csr"},
    { 'name' => 'private-key', 'file' => $httpdConfDir . "/ssl.key/server.key"}
  );
  my @certFileNodes;
  for my $certInfo (@certInfoMap) {
    my $certificate = DumperUtils::readFile($certInfo->{'file'});
    next unless (defined $certificate);
    push @certFileNodes, XmlNode->new($certInfo->{'name'}, 'content' => $certificate);
  }
  unless (@certFileNodes) {
    return undef;
  }

  return XmlNode->new('certificates',
    'children' => [
      XmlNode->new('certificate',
        'attributes' => {
          'name' => $domainName
        },
        'children' => \@certFileNodes
      )
    ]
  );
}

sub _getWebUsers {
  my ($domainName, $ssi, $cgi, $perl, $frontpage) = @_;

  my @result;
  my $webUsers = Dumper::getWebUsers($domainName);
  while (my ($username, $password) = each (%$webUsers)) {
    my $contentNode = ContentDumper::getWebUserSiteContent($contentDumperBase, $domainName, $username);
    # if a user has no web content, it is not a web user, skip it
    next unless (ref($contentNode) =~ /XmlNode/);

    my $webUserNode = XmlNode->new('webuser',
      'attributes' => {
        'name' => $username
      },
      'children' => [
        $contentNode,
        CommonXmlNodes::sysuser($username, $password, "/web_users/$username"),
        _makeWebUserScriptingNode($ssi, $cgi, $perl, $frontpage)
      ]
    );
    push @result, $webUserNode;
  }

  return \@result;
}

sub _getDatabases {
  my ($domainName) = @_;

  Logging::trace("Getting databases for domain '$domainName' ...");

  my @userDatabases = Dumper::getDatabaseNames($domainName);
  if (!@userDatabases) {
    return;
  }

  my $databasesNode = XmlNode->new('databases');
  foreach my $dbName (@userDatabases) {
    $databasesNode->addChild(_getDatabaseNode(Dumper::getDbAdminCredentials(), $dbName, $domainName));
  }

  return $databasesNode;
}

sub _getDatabaseNode {
  my ($dbAdminLogin, $dbAdminPassword, $dbName, $domainName) = @_;

  my $dbType = 'mysql';

  my $dbNode = XmlNode->new('database');
  $dbNode->setAttribute('name', $dbName);
  $dbNode->setAttribute('type', $dbType);
  $dbNode->setAttribute('version', Db::MysqlUtils::getVersion());

  my $contentNode = DatabaseContentDumper::getDatabaseContent(
    $contentDumperBase, $dbAdminLogin, $dbAdminPassword, $dbName, $dbType, $domainName);
  $dbNode->addChild($contentNode) if defined $contentNode;

  my $dbCon = Db::DbConnect::getDbConnect($dbType, $dbAdminLogin, $dbAdminPassword, 'mysql');
  # Skip 'root' user during DB users dump.
  Db::DbConnect::addDbUsers($dbNode, $dbName, $dbCon, ['root']);
  $dbCon->{'DISCONNECT'}->();

  return $dbNode;
}

sub _getSiteIPInfo {
  my ($siteId) = @_;

  my %ipinfo = Dumper::getIpInfoForSite($siteId);

  unless ( $ipinfo{'enabled'} ) {
    # It means that ip information is not available (maybe configuration file is absent)
    return;
  }

  if ($ipinfo{'namebased'} eq '1') {
    $ipinfo{'nbaddrs'} =~ /\[\'(.+)\'\]/;
    return {'ip' => $1, 'type' => 'shared'};
  } else {
    $ipinfo{'ipaddrs'} =~ /\[\'(.+)\'\]/;
    return {'ip' => $1, 'type' => 'exclusive'};
  }
}

sub _getMailSystem {
  my ($domainName, $siteUsersInfo) = @_;

  Logging::trace("Getting emails for domain '$domainName' ...");

  my %sendmailInfo = Dumper::getSpecificInfoForSite(Dumper::getSiteIdByDomain($domainName), 'sendmail');
  unless ($sendmailInfo{'enabled'}) {
    return;
  }

  my $aliases = Dumper::getMailAliases(
    $domainName, 
    # maillist aliases are removed as some of them are recreated by Plesk when creating maillists
    Dumper::getMaillistsAliases($domainName)
  );

  # - simple alias is an alias that has only one local delivery address
  # it can be mapped directly to an e-mail alias
  # - complex aliases require to have a separate mailuser
  my ($simpleAliases, $complexAliases) = _splitHash(
    $aliases, 
    sub { 
      my ($alias) = @_;
      return @{$alias->{'responders'}} == 0 && @{$alias->{'localDelivery'}} == 1 && @{$alias->{'redirects'}} == 0;
    } 
  );

  Logging::debug("Simple e-mail aliases for domain '$domainName: " . join(', ', keys(%{$simpleAliases})));
  Logging::debug("Complex e-mail aliases for domain '$domainName: " . join(', ', keys(%{$complexAliases})));

  my $simpleAliasesByUserName = _makeSimpleAliasesByUserNameHash($simpleAliases);

  my $mailSystemNode = XmlNode->new('mailsystem', 'children' => [
    # TODO what status should mailsystem have ? maybe domain's one ?
    XmlNode->new('properties', 'children' => [CommonXmlNodes::status(1, 'admin')])
  ]);

  my $mailUsersNode = XmlNode->new('mailusers');
  while (my ($userName, $userInfo) = each %{$siteUsersInfo}) {
    my $userAliases = $simpleAliasesByUserName->{$userName};
    unless (defined($userAliases)) {
      $userAliases = [];
    }
    $mailUsersNode->addChild(_getMailUserNode($domainName, $userName, $userInfo, $userAliases));
  }

  while (my ($alias, $aliasInfo) = each %{$complexAliases}) {
    $mailUsersNode->addChild(_getAliasMailUserNode($domainName, $alias, $aliasInfo));
  }

  $mailSystemNode->addChild($mailUsersNode);

  $mailSystemNode->addChild(XmlNode->new('preferences', 'children' => [
    # TODO investigate
    XmlNode->new('catch-all', 'content' => 'reject'),
    XmlNode->new('web-mail', 'content' => 'horde'),
    XmlNode->new('grey-listing', 'content' => 'on')
  ]));

  return $mailSystemNode;
}


sub _getAliasMailUserNode {
  my ($domainName, $alias, $aliasInfo) = @_;
    
  my @autoresponderNodes = map { _getAutoresponderNode($_->{'subject'}, $_->{'body'}) } @{$aliasInfo->{'responders'}};
  my $autorespondersNode = XmlNode->new(
    'autoresponders', 
    'children' => \@autoresponderNodes 
  );

  my @forwardEmails = (
    (map { $_ . '@' . $domainName } @{$aliasInfo->{'localDelivery'}}),
    @{$aliasInfo->{'redirects'}}
  );

  my @forwardingNodes = map { XmlNode->new('forwarding', 'content' => $_) } @forwardEmails;
  
  my $mailUserNode = XmlNode->new('mailuser',
    'attributes' => {
      'name' => $alias,
      'forwarding-enabled' => @forwardEmails ? 'true' : 'false'
    },
    'children' => [
      _getMailuserPropertiesNode(),
      new XmlNode('preferences',
        'children' => [
          @forwardingNodes,
          $autorespondersNode
        ]
      )
    ]
  );

  return $mailUserNode;
}

sub _getAutoresponderNode {
  my ($subject, $text) = @_;
  
  return XmlNode->new(
    'autoresponder',
    'children' => [
      XmlNode->new('text', 'content' => EncodeBase64::encode($text)),
    ],
    'attributes' => {
      'subject' => EncodeBase64::encode($subject)
    }
  );
}

sub _splitHash {
  my ($hash, $conditionFunction) = @_;
  my (%hashTrue, %hashFalse);

  while (my ($key, $value) = each %{$hash}) {
    if ($conditionFunction->($value)) {
      $hashTrue{$key} = $value;
    } else {
      $hashFalse{$key} = $value;
    }
  } 

  return (\%hashTrue, \%hashFalse);
}

sub _makeSimpleAliasesByUserNameHash {
  my ($simpleAliases) = @_;
  my %result;

  while (my ($alias, $aliasInfo) = each %{$simpleAliases}) {
    my $email = $aliasInfo->{'localDelivery'}->[0];
    unless (defined($result{$email})) {
      $result{$email} = []; 
    }
    push(@{$result{$email}}, $alias);
  }

  return \%result;
}

sub _getMailUserNode {
  my ($domainName, $userName, $userInfo, $aliases) = @_;

  my $mailUserNode = XmlNode->new('mailuser',
    'attributes' => {
      'name' => $userName,
      # TODO investigate this attribute
      'forwarding-enabled' => 'false'},
    'children' => [
      _getMailuserPropertiesNode($userInfo->{'password'})
    ]
  );

  my $mailBoxNode = XmlNode->new('mailbox',
    'attributes' => {'type' => 'mdir', 'enabled' => 'true'},
    'children' => [ContentDumper::getMailBoxContent($contentDumperBase, $domainName, $userName, $userInfo)]);

  my @aliasNodes = map { XmlNode->new('alias', 'content' => $_) } @{$aliases};

  my $domainSiteDir = Dumper::getFilesystemByDomainName($domainName);
  my $spamassassinNode = _makeSpamassassinNode("$domainSiteDir/home/$userName/.spamassassin/user_prefs", "$domainSiteDir/home/$userName");
  if (!defined($spamassassinNode)) {
      # use default site/domain settings
      $spamassassinNode = _makeSpamassassinNode("$domainSiteDir/etc/mail/spamassassin/local.cf", $domainSiteDir);
  }

  $mailUserNode->addChild(XmlNode->new('preferences', 'children' => [$mailBoxNode, @aliasNodes, $spamassassinNode ]));

  return $mailUserNode;
}

sub _getMailuserPropertiesNode {
  my ($userPassword) =  @_;
  return XmlNode->new('properties',
    'children' => [ 
      defined($userPassword) ? CommonXmlNodes::encryptedPassword($userPassword) : CommonXmlNodes::emptyPassword()
    ]
  );
}

sub _getMaillists {
  my ($domainName) = @_;

  my $maillists = Dumper::getMaillists($domainName);

  my $maillistsNode = XmlNode->new('maillists',
    'children' => [
      XmlNode->new('properties',
        'children' => [ CommonXmlNodes::status(keys %{$maillists} > 0, 'admin') ]
      )
    ]
  );

  while (my ($maillistName, $maillistInfo) = each %{$maillists}) {
    $maillistsNode->addChild(_getMaillistNode($maillistName, $maillistInfo));
  }

  return $maillistsNode;
}

sub _getMaillistNode {
  my ($maillistName, $maillistInfo) = @_;

  return XmlNode->new('maillist',
    'attributes' => {
      'name' => $maillistName
    },
    'children' => [
      CommonXmlNodes::status(1, 'admin'),
      XmlNode->new('owner', 'content' => $maillistInfo->{'owner'}),
      _makePlainPasswordNode($maillistInfo->{'adminPassword'}),
      map { XmlNode->new('recipient', 'content' => $_); } @{$maillistInfo->{'members'}}
    ]
  );
}

sub _makeSpamassassinNode {
  my ($configFile, $configHomeDir) = @_;
  use constant DEFAULT_SPAMASSASSIN_REQUIRE_SCORE => 5;

  unless (-r "$configFile") {
      return undef;
  }

  my $spamassassinConfig = SpamAssassinCfg::parseConfig($configFile, $configHomeDir);
  return undef unless defined($spamassassinConfig);

  my $spamassassinConfigRequireScore = SpamAssassinCfg::getConfigRequireScore($spamassassinConfig);
  my $spamassassinConfigRewriteHeaderArray = SpamAssassinCfg::getConfigRewriteHeadr($spamassassinConfig);
  my $spamassassinConfigRewriteHeaderText;
  my $spamassassinConfigAction = Dumper::getProcmailrcAction("$configHomeDir/.procmailrc");

  if (defined($spamassassinConfigRewriteHeaderArray) && @{$spamassassinConfigRewriteHeaderArray} == 2) {
    my ($argumentRewrite, $textRewrite) = @{$spamassassinConfigRewriteHeaderArray};

    $spamassassinConfigRewriteHeaderText = $textRewrite if ($argumentRewrite =~ m/subject/i);
  }

  if (!defined($spamassassinConfigRewriteHeaderText) && $spamassassinConfigAction eq "mark") {
      return undef;
  }
  $spamassassinConfigRequireScore = DEFAULT_SPAMASSASSIN_REQUIRE_SCORE unless defined($spamassassinConfigRequireScore);

  my ($spamassassinNode) = XmlNode->new('spamassassin',
    'attributes' => {
            'status'    => 'on',
            'hits'      => $spamassassinConfigRequireScore,
            'action'    => $spamassassinConfigAction,
    },
    'children'   => [
        _makeSpamassassinListNodes('blacklist-member', SpamAssassinCfg::getConfigBlackList($spamassassinConfig)),
        _makeSpamassassinListNodes('whitelist-member', SpamAssassinCfg::getConfigWhiteList($spamassassinConfig)),
        _makeSpamassassinListNodes('unblacklist-member', SpamAssassinCfg::getConfigUnBlackList($spamassassinConfig)),
        _makeSpamassassinListNodes('unwhitelist-member', SpamAssassinCfg::getConfigUnWhiteList($spamassassinConfig))
    ]
  );
  if (defined($spamassassinConfigRewriteHeaderText)) {
      $spamassassinNode->setAttribute('subj-text', $spamassassinConfigRewriteHeaderText);
  }

  return $spamassassinNode;
}

sub _makeSpamassassinListNodes {
    my ($nodeName, $contentArrayPtr) = @_;

    return undef unless defined($contentArrayPtr);

    return map { XmlNode->new($nodeName, 'content' => $_) } @{$contentArrayPtr};
}

use constant RESOURCE_UNLIMITED => -1;

sub _getResellerLimits {
  my ( $resellerId ) = @_;

  my %limits = Dumper::getResellerLimits($resellerId);

  my %convertedLimits = _convertLimitsHash( 
    \%limits,
    { 
      'disk_space' => 'diskquota',
      # maximum number of users is mapped to maximum number of mailboxes
      # as there is no exact mapping 
      # and every user in PPCPL has exactly one mailbox
      'max_box' => 'users',
      'max_traffic' => 'bandwidth'
    }
  );

  if (defined($limits{'ip_based_sites'}) && defined($limits{'name_based_sites'})) {
    $convertedLimits{'max_cl'} = $limits{'ip_based_sites'} + $limits{'name_based_sites'};
  } else {
    $convertedLimits{'max_cl'} = RESOURCE_UNLIMITED;
  }

  return %convertedLimits;
}

sub _getResellerPermissions {
  my ( $resellerId ) = @_;

  return (
    'manage_dns' => Dumper::canResellerManageDNS($resellerId) eq '1',
    'create_clients' => '1', # PPCPL reseller is able to create client
    'create_domains' => '1'  # with domain
  );
}

sub _getDomainLimits {
  my ( $domainName ) = @_;

  my $siteId = Dumper::getSiteIdByDomain($domainName);

  my %limits = Dumper::getSiteLimits($siteId);
  
  my %convertedLimits = _convertLimitsHash( 
    \%limits,
    { 
      'disk_space' => 'diskquota',
      'max_box' => 'users',
      'max_traffic' => 'bandwidth'
    }
  );

  $convertedLimits{'max_site'} = '1';

  return %convertedLimits;
}

sub _convertLimitsHash {
  my ($srcHash, $map) = @_;

  my %result;

  while (my ($dstKey, $srcKey) = each %{$map}) {
    if (defined($srcHash->{$srcKey})) {
      $result{$dstKey} = $srcHash->{$srcKey}
    } else {
      $result{$dstKey} = RESOURCE_UNLIMITED; 
    }
  }

  return %result;
}

sub _getDomainPermissions {
  my ( $domainName ) = @_;
  
  my $siteId = Dumper::getSiteIdByDomain($domainName);
  
  my %bindInfo = Dumper::getBindInfoForSite($siteId);
  
  return (
    'manage_dns' => $bindInfo{'zone_mgmt'} eq '1'
  );
}

sub _makePlainPasswordNode {
  my ($password) = @_;

  return XmlNode->new('password',
    'attributes' => {'type' => 'plain'},
    'content' => $password
  );
}

sub _makeHostingPreferencesNode {
  my ($domainName, $usersInfoMapPtr) = @_;

  my $siteId = Dumper::getSiteIdByDomain($domainName);
  my %siteInfo = Dumper::getSpecificInfoForSite($siteId, 'siteinfo');
  my $siteAdminLogin = $siteInfo{'admin_user'};

  return
    XmlNode->new('preferences', 
      'children' => [
        CommonXmlNodes::sysuser($siteAdminLogin, $usersInfoMapPtr->{$siteAdminLogin}{'password'}, "/var/www/vhosts/$domainName"),
        _makeLogrotationNode($domainName, $siteAdminLogin),
        _makeAnonFtpNode($domainName, $siteId),
        @{_makePDirNodes($domainName)},
      ]
    );
}

sub _makeLogrotationNode {
  my ($domainName, $siteAdminLogin) = @_;

  my %settings = Dumper::getLogrotateSettings($domainName) or return;

  my $rotationNode;
  if (defined $settings{'size'} && $settings{'size'} != 0) {
    $rotationNode = XmlNode->new('logrotation-maxsize',
      'content' => $settings{'size'}
    );
  } else {
    $rotationNode = XmlNode->new('logrotation-period',
      'attributes' => {
        'period' => $settings{'period'}
      }
    );
  }

  my $logrotateNode = XmlNode->new('logrotation',
    'attributes' => {
      'enabled' => 'true',
      'max-number-of-logfiles' => $settings{'number-of-rotated-files'},
      'compress' => 'true',
      'email' => "$siteAdminLogin". "@". "$domainName"
    },
    'children' => [
      $rotationNode
    ]
  );

  return $logrotateNode;
}

sub _makeAnonFtpNode {
  my ($domainName, $siteId) = @_;

  my %ipinfo = Dumper::getSpecificInfoForSite($siteId, 'ipinfo');
  my %anonftpInfo = Dumper::getSpecificInfoForSite($siteId, 'anonftp');
  my %proftpdInfo = Dumper::getSpecificInfoForSite($siteId, 'proftpd');

  my $anonFtpContent = ContentDumper::getAnonFtpContent($contentDumperBase, $domainName);

  # a warning should be issued for each incompletely migrated anonymous FTP

  if ($anonftpInfo{'enabled'} && $ipinfo{'namebased'}) {
    PreMigration::message('ANONYMOUS_FTP_ON_SHARED_IP', {'domain' => $domainName});
    PreMigration::assert($anonFtpContent, 'ANONYMOUS_FTP_DIRECTORY_CONTENT', {'domain' => $domainName});
  }

  # we need anonftp access enabled when it's enabled in PPCPL,
  # but since Plesk complains that it can't enable anonftp for name-based sites,
  # let's ask for anonftp access enabled only if site is ip-based.
  my $enableAnonftp = !$ipinfo{'namebased'} && $anonftpInfo{'enabled'};

  # make anonftp node if we need to migrate anonftp content, or to enable anonftp access, or both.
  if ($anonFtpContent || $enableAnonftp) {
    return XmlNode->new('anonftp',
      'attributes' => {
        'pub' => $enableAnonftp ? 'true' : 'false',
        'incoming' => $enableAnonftp ? 'true' : 'false',
        'display-login' => 'false'
      },
      'children' => [
        $anonFtpContent
      ]
    );
  } else {
    return undef;
  }
}

sub _makePDirNodes {
  my ($domainName) = @_;

  my @result;
  my @pDirInfoList = Dumper::getProtectedDirectories($domainName);
  for my $pDirInfo_ptr ( @pDirInfoList ) {
    my $pDirNode = XmlNode->new('pdir',
      'attributes' => {
        'name' => $pDirInfo_ptr->{'name'},
        'nonssl' => $pDirInfo_ptr->{'cgi'} eq 'true' ? 'false' : 'true',
        'cgi' => $pDirInfo_ptr->{'cgi'}
      }
    );
    $pDirNode->setAttribute('title', $pDirInfo_ptr->{'title'}) if $pDirInfo_ptr->{'title'};
    my %usersMap = %{$pDirInfo_ptr->{'users'}};
    for my $username ( keys %usersMap ) {
      $pDirNode->addChild(XmlNode->new('pduser',
        'attributes' => {
          'name' => $username
        },
        'children' => [
          CommonXmlNodes::encryptedPassword($usersMap{$username})
        ]
      ));
    } # for each user
    push @result, $pDirNode;
  } # for each protected directory

  return \@result;
}

sub _makePreferencesNode {
  my ( $email ) = @_;

  return XmlNode->new('preferences', (
    'children' => [
      XmlNode->new('pinfo', (
        'attributes' => { 'name' => 'email' },
        'content' => $email 
      ))
    ]
  ));
}

sub _makePHostingScriptingNode {
  my ($ssi, $cgi, $perl, $mm4, $mm5, $frontpage) = @_;

  my $result = _makeBasicScriptingNode($ssi, $perl);
  $result->setAttribute('php_handler_type', 'isapi');
  $result->setAttribute('php_isapi', 'true');
  $result->setAttribute('php_safe_mode', 'false');
  $result->setAttribute('cgi', $cgi->{'enabled'} ? 'true' : 'false');
  $result->setAttribute('asp_dot_net', 'false');
  $result->setAttribute('coldfusion', 'false');
  $result->setAttribute('miva', $mm4->{'enabled'} || $mm5->{'enabled'} ? 'true' : 'false');
  $result->setAttribute('fp', $frontpage->{'enabled'} ? 'true' : 'false'); # enable FrontPage
  $result->setAttribute('fp_enable', $frontpage->{'enabled'} ? 'true' : 'false'); # enable FrontPage authoring

  return $result;
}

sub _makeWebUserScriptingNode {
  my ($ssi, $cgi, $perl, $frontpage) = @_;

  my $result = _makeBasicScriptingNode($ssi, $perl);
  $result->setAttribute('php_isapi', 'false');
  $result->setAttribute('cgi', $cgi->{'enabled'} ? 'true' : 'false');
  $result->setAttribute('asp_dot_net', 'false');
  $result->setAttribute('fp_enable', $frontpage->{'enabled'} ? 'true' : 'false');

  return $result;
}

sub _makeSubdomainScriptingNode {
  my ($subdomainInfo, $ssi, $perl, $mm4, $mm5) = @_;

  my $result = _makeBasicScriptingNode($ssi, $perl);
  $result->setAttribute('miva', $mm4->{'enabled'} || $mm5->{'enabled'} ? 'true' : 'false');
  $result->setAttribute('php_handler_type', 'isapi');
  $result->setAttribute('coldfusion', 'false');
  $result->setAttribute('cgi', (
            (defined($subdomainInfo->{cgi}) && "_$subdomainInfo->{cgi}" eq "_1") ? "true" : "false"));

  return $result;
}

sub _makeBasicScriptingNode {
  my ($ssi, $perl) = @_;

  return XmlNode->new('scripting',
    'attributes' => {
      'ssi'       => $ssi->{'enabled'} ? 'true' : 'false',
      'php'       => 'true',
      'perl'      => $perl->{'enabled'} ? 'true' : 'false',
      'asp'       => 'false',
      'python'    => 'false',
      'fastcgi'   => 'true'
    }
  );
}

sub _makeDnsZoneNode {
  my ( $siteId ) = @_;
    
  my %siteInfo = Dumper::getSpecificInfoForSite($siteId, 'siteinfo');
  my %dnsInfo  = Dumper::getSpecificInfoForSite($siteId, 'bind');
  my $domainName = $siteInfo{'domain'};

  XmlNode->new('dns-zone', (
    'attributes' => {
        'email'         => $siteInfo{'email'},
        'serial-format' => 'YYYYMMDDNN',
        'type'          => 'master',
    },
    'children' => [
        # translate PPCPL "Add to DNS" setting to Plesk 'enabled'/'disabled'
        CommonXmlNodes::status($dnsInfo{'enabled'}, 'admin'),
        _makeDnsZoneProperties($domainName)
    ] 
  ));
}

sub _makeDnsZoneProperties {
  my ( $domainName ) = @_;
  my @records = Dumper::getDnsZoneProperties($domainName);
  my @dnsZoneNodes = ();

  my $soaRecord = shift @records;
  push @dnsZoneNodes, _makeDnsZoneParams(@$soaRecord);
  
  for my $rec (@records) {
    push @dnsZoneNodes, CommonXmlNodes::dnsRecord(@{$rec});
  }
  return @dnsZoneNodes;
}

# parse SOA record, create <dns-zone-param> elements
sub _makeDnsZoneParams {
  my ($src, $ttl, $in, $type, $ns, $email, $serial, @values) = @_;
  my @parameters = qw(ttl refresh retry expire minimum);
  my @records;

  unshift @values, $ttl;
  if ('SOA' eq $type && @parameters == @values) {
    Logging::trace("DNS SOA record: " .  join ' ', @_);
    for my $value (@values) {
      my $record = XmlNode->new('dns-zone-param',
            'attributes' => {
                'name'  => shift @parameters,
                'unit'  => 'second',
                'value' => $value
            });
      push @records, $record;
    }
  }
  return @records;
}

1;
