# Copyright 1999-2012. Parallels IP Holdings GmbH. All Rights Reserved.
package ConfixxConfig;

# 'use strict' is not used here, as we execute Confixx config file, which is a regular Perl script
# with variables that don't have 'my' in their declarations
use warnings;

use Logging;

my $configLoaded = 0;

sub getValue {
  my ($varName) = @_;

  unless ($varName) {
    Logging::warning("Attempt to get Confixx config variable by undefined name");
    return undef;
  }

  unless ($configLoaded) {
    _loadMainConfig();
  }

  # at this point, either config is loaded or program is terminated, don't check it again

  return eval "\$$varName";
}

sub _loadMainConfig {
  if (!$configLoaded) {
    my $path = _getConfixxMainConfigPath();
    if (!defined($path)) {
      Logging::error("Confixx config was not found");
      die; # absence of config is critical error, don't go any further
    }

    Logging::debug("Found Confixx config at $path, loading");

    do $path;

    $configLoaded = 1;
  }

  return 1;
}

sub _getConfixxMainConfigPath() {
  return 
    _guessConfixxMainConfigPathFromEnv() ||
    _guessConfixxMainConfigPathFromStandardPaths() ||
    _guessConfixxMainConfigPathFromCrontab();
}

sub _guessConfixxMainConfigPathFromEnv {
  if (exists($ENV{'CONFIXX_ROOT'} ) && -d $ENV{'CONFIXX_ROOT'}) {
    my $path = $ENV{'CONFIXX_ROOT'} . '/confixx_main.conf';
    return $path if -r $path;
  }

  return undef;
}

sub _guessConfixxMainConfigPathFromStandardPaths {
  foreach my $prefix ('./','/root/confixx/','/usr/local/confixx/','/confixx/') {
    my $path = $prefix . 'confixx_main.conf';
    return $path if -r $path;
  }

  return undef;
}

sub _guessConfixxMainConfigPathFromCrontab {
  return 
    _guessConfixxMainConfigPathFromCrontabCommand(AgentConfig::grepBin() . ' -hr confixx /etc/crontab /etc/cron.d') ||
    _guessConfixxMainConfigPathFromCrontabCommand(AgentConfig::crontabBin() . ' -l');
}

sub _guessConfixxMainConfigPathFromCrontabCommand {

  my ($crontabLinesCommand) = @_;

  foreach my $line (`$crontabLinesCommand`) {
    chomp $line;
    next if ( $line =~ /^\s*#/ ); # skip comment line

    my @lineParts = split(' ', $line);
    my $command = $lineParts[scalar(@lineParts) - 1]; # we assume that the last part of a line is a command
    if ($command =~ /^(.*)\/[^\/]*$/) {
      my $commandBaseDir = $1;
      my $path = $commandBaseDir. '/confixx_main.conf';

      return $path if -r $path;
    }
  }

  return undef;
}

1;
