#!/usr/bin/perl

use strict;
use warnings;

use lib '.';
use lib 'shared';
use lib 'shared_legacy';

use Carp;

use LegacyPanel::Tree::Builder;
use LegacyPanel::Tree::Util;
use LegacyPanel::EntityFactory;
use LegacyPanel::PreMigrationMessage;
use LegacyPanel::PreMigrationReport;
use LegacyPanel::Toolbox;

# Disable printing all log messages, so customer would see only the pre-migration report.
# Do it before other modules are initialized, so all initialization logging messages are not printed.
use Logging;
BEGIN {
  Logging::setVerbosity(0);
}

use Dumper;
use PreMigrationMessages;

sub main () {

  my $entityFactory = LegacyPanel::EntityFactory->new(
    'admin'     => 'EnsimAdmin',
    'reseller'  => 'EnsimReseller',
    'client'    => 'EnsimClient',
    'domain'    => 'EnsimDomain',
  );

  my @messageList = ();
  my $tree = LegacyPanel::Tree::Builder::buildTree($entityFactory);
  push @messageList, preMigrationCheckOverall($tree);
  push @messageList, preMigrationCheckObjects($tree);

  print LegacyPanel::PreMigrationReport::makeReport(\@messageList);
}

sub preMigrationCheckOverall($) {
  my ($tree) = @_;
  my $defaultObject = 'source server';

  unless (-e "/usr/bin/getapplversion") {
    print "Can't locate the PPCPL on this host.\n";
    exit (1);
  }

  my @messageList = ();

  my $diskOccupied = preMigrationGetDiskspace($tree);
  my $printDiskOccupied = sprintf("%.3f", $diskOccupied / 1024.0 / 1024.0 / 1024.0);
  push @messageList, createMessage($defaultObject, undef, 'DISKSPACE', 'diskspace' => $printDiskOccupied);

  my ($apacheUid, $apacheGid) = LegacyPanel::Toolbox::getApacheUid();
  if (((not defined($apacheUid)) or (not defined($apacheGid)))) {
    push @messageList, createMessage($defaultObject, undef, 'APACHE_UID_UNKNOWN');
  }

  if (Dumper::getLocalNamedStatus() != 1) {
    push @messageList, createMessage($defaultObject, undef, 'NAMED_NOT_RUNNING');
  }

  push @messageList, createMessage($defaultObject, undef, 'MAILQUOTA_IS_APPROXIMATED');

  return @messageList;
}

sub preMigrationCheckObjects($) {
  my ($tree) = @_;
  return LegacyPanel::Tree::Util::applyVisitor($tree, &preMigrationCheckNodeFilter);
}

sub preMigrationCheckNodeFilter($) {
  my ($tree) = @_;

  my @messages = ();

  my $t = $tree->getNodeValue();
  if ($t->isa('LegacyPanel::Entity::Domain')) {
    if (scalar(grep {$_ eq 'admin'} $t->getDatabaseUsers()) > 0) {
      push @messages, createMessage("DB user 'admin'", $tree, 'DB_USER_ADMIN');
    }
  } elsif ($t->isa('LegacyPanel::Entity::Reseller')) {
    if ($t->getUsername() eq 'admin') {
      push @messages, createMessage(undef, $tree, 'DB_RESELLER_ADMIN');
    }
  }
  return @messages;
}

sub preMigrationGetDiskspace($) {
  my ($tree) = @_;
  my @diskspaces = LegacyPanel::Tree::Util::applyVisitor($tree, &domainDiskspaceFilter);
  my $sum = 0;
  foreach my $i (@diskspaces) {
    $sum += $i;
  }
  return $sum;
}
  
sub domainDiskspaceFilter($) {
  my ($tree) = @_;

  my $t = $tree->getNodeValue();
  if ($t->isa('LegacyPanel::Entity::Domain')) {
    return $t->getConsumedDiskSpace();
  }
  return ();
}

sub createMessage($$$%) {
  my ($object, $carrier, $messageId, %parameters) = @_;

  if (exists($PreMigrationMessages::MESSAGES{$messageId})) {
    my %message = %{$PreMigrationMessages::MESSAGES{$messageId}};

    # substitute parameters
    if (scalar(%parameters)) {
      foreach my $part ('message', 'resoultion') {
        if (exists($message{$part}) and defined($message{$part})) {
          foreach my $key (keys %parameters) {
            $message{$part} =~ s/{$key}/$parameters{$key}/g;
          }
        }
      }
    }

    return LegacyPanel::PreMigrationMessage->new(
      'object' => $object,
      'carrier' => $carrier,
      %message,
    );
  }
  return undef;
}

main();
