<!-- Copyright 1999-2015. Parallels IP Holdings GmbH. All Rights Reserved. -->
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
  <xsl:output indent="yes" encoding="UTF-8"/>
  <xsl:variable name='agent_name' select='RaQ550dump/@agent-name'/>

  <xsl:template match="RaQ550dump">
    <migration-dump agent-name="{$agent_name}">
      <xsl:for-each select="client">
        <xsl:call-template name="create-client"/>
      </xsl:for-each>
    </migration-dump>
  </xsl:template>

  <xsl:template match="@ip">
    <ip>
      <xsl:choose>
        <xsl:when test="../anonftp">
          <ip-type>exclusive</ip-type>
        </xsl:when>
        <xsl:otherwise>
          <ip-type>shared</ip-type>
        </xsl:otherwise>
      </xsl:choose>
      <ip-address>
        <xsl:value-of select='.'/>
      </ip-address>
    </ip>
  </xsl:template>
  
  <!--
  Key used for removal of duplicate IP addresses in client's pool:
  [IP address value][name of the client][indication of ip based hosting]
  -->
  <xsl:key name="ip" match="domain/@ip" use="concat(., ../../sysuser/@name, ../anonftp)"/>

  <xsl:template name="create-client">
    <client>
      <xsl:attribute name="name">
        <xsl:call-template name="mangle-sysuser-name">
          <xsl:with-param name="name" select="sysuser/@name"/>
        </xsl:call-template>
      </xsl:attribute>

      <xsl:if test="@fullname">
        <xsl:attribute name="contact"><xsl:value-of select="@fullname"/></xsl:attribute>
      </xsl:if>

      <xsl:copy-of select="sysuser/password"/>

      <xsl:choose>
        <xsl:when test="@enabled='true'">
		  <status><enabled/></status>
        </xsl:when>
        <xsl:otherwise>
		  <status><disabled-by name="admin"/></status>
        </xsl:otherwise>
      </xsl:choose>

      <limit name="disk_space"><xsl:value-of select="@disk-quota"/></limit>

      <ip_pool>
        <xsl:apply-templates select="domain/@ip[generate-id(.) = generate-id(key('ip',concat(., ../../sysuser/@name, ../anonftp)))]"/>
      </ip_pool>

      <xsl:for-each select="domain">
        <xsl:call-template name="create-domain"/>
      </xsl:for-each>
    </client>
  </xsl:template>

  <xsl:template name="create-domain">
	<xsl:variable name="domainname" select="@name"/>

	<domain name="{@name}" www="true">
	  <xsl:apply-templates select="@ip"/>

      <!-- STATUS -->
      <xsl:choose>
        <xsl:when test="@state='false'">
		  <status><disabled-by name="admin"/></status>
        </xsl:when>
        <xsl:otherwise>
		  <status><enabled/></status>
        </xsl:otherwise>
      </xsl:choose>

      <!-- LIMIT -->
      <xsl:for-each select="limit">
        <xsl:if test="@name='disk-quota'">
          <xsl:call-template name="limit">
            <xsl:with-param name="limit" select="'disk_space'"/>
            <xsl:with-param name="value" select="text()"/>
          </xsl:call-template>
        </xsl:if>
      </xsl:for-each>

      <!-- MAILSYSTEM -->
      <xsl:call-template name="create-mailsystem"/>

      <!-- DNSRECS -->
<!-- Removed as still not supported by Plesk
      <xsl:for-each select="email-server-alias">
        <dnsrec type="MX" src="{text()}" dst="{$domainname}" opt="10"/>
      </xsl:for-each>
      <xsl:for-each select="web-server-alias">
        <dnsrec type="PTR" src="{text()}" dst="{$domainname}"/>
      </xsl:for-each> -->

      <!-- DOMAIN ALIAS -->
      <xsl:copy-of select="domain-alias"/>

      <!-- MAILLIST -->
	  <xsl:if test="maillist">
		<maillists>
		  <status><enabled/></status>
		  <xsl:apply-templates select="maillist"/>
		</maillists>
	  </xsl:if>

      <!-- TODO TRAFFIC -->

      <!-- CERTIFICATE -->
      <xsl:copy-of select="certificate"/>

	  <!-- DOMAINUSER -->
	  <xsl:if test="@admin-user-name">
		<domainuser>
		  <xsl:variable name="admin-user" select="user[sysuser/@name = current()/@admin-user-name]"/>
		  <xsl:if test="$admin-user/@fullname">
			<xsl:attribute name="contact"><xsl:value-of select="$admin-user/@fullname"/></xsl:attribute>
		  </xsl:if>
		  <xsl:choose>
			<xsl:when test="$admin-user/@state = 'false'">
			  <xsl:attribute name="status">false</xsl:attribute>
			</xsl:when>
			<xsl:otherwise>
			  <xsl:attribute name="status">true</xsl:attribute>
			</xsl:otherwise>
		  </xsl:choose>
		  <xsl:copy-of select="$admin-user/sysuser/password"/>
		  <pinfo name="email">
		    <xsl:value-of select="concat($admin-user/mail/@mailname, '@', @name)"/>
		  </pinfo>
		  <permission name="multiple-sessions" allowed="true"/>
		</domainuser>
	  </xsl:if>

      <!-- PHOSTING -->
      <phosting cid_docroot="{@cid}" fpauth="false">
        <xsl:call-template name="convert-boolean-permission">
          <xsl:with-param name="raqname" select="'ssl'"/>
          <xsl:with-param name="pleskname" select="'https'"/>
        </xsl:call-template>
        <xsl:call-template name="convert-boolean-permission">
          <xsl:with-param name="raqname" select="'frontpage'"/>
          <xsl:with-param name="pleskname" select="'fp'"/>
        </xsl:call-template>

        <xsl:choose>
          <xsl:when test="permission[@name='frontpage'] and permission[@name='ssl']">
            <xsl:attribute name="fpssl">true</xsl:attribute>
          </xsl:when>
          <xsl:otherwise>
            <xsl:attribute name="fpssl">false</xsl:attribute>
          </xsl:otherwise>
        </xsl:choose>

		<xsl:choose>
		  <xsl:when test="@admin-user-name">
			<xsl:copy-of select="user[sysuser/@name = current()/@admin-user-name]/sysuser"/>
		  </xsl:when>
		  <xsl:otherwise>
			<sysuser name="admin{position()}"><password type="plain"/></sysuser>
		  </xsl:otherwise>
		</xsl:choose>

        <!-- casp permission ignored -->

        <!-- SCRIPTING -->
        <xsl:call-template name="create-scripting"/>

        <xsl:if test="limit[@name='frontpage'] and fpuser">
          <fpuser name="admin{position()}">
            <xsl:copy-of select="fpuser/password"/>
          </fpuser>
        </xsl:if>

        <!-- ANONFTP -->
        <xsl:apply-templates select="anonftp"/>

        <!-- WEBUSER -->
        <xsl:apply-templates select="user[sysuser/@name != current()/@admin-user-name]" mode="webuser"/>

      </phosting>
    </domain>
  </xsl:template>

  <xsl:template match="maillist">
    <maillist name="{@name}">
      <status><enabled/></status>
	  <owner><xsl:value-of select="@owner"/></owner>
      <xsl:copy-of select="password"/>
      <xsl:copy-of select="recipient"/>
    </maillist>
  </xsl:template>

  <xsl:template name="create-mailsystem">
    <mailsystem>
	  <status><enabled/></status>
      <xsl:apply-templates select="user" mode="mailuser"/>
    </mailsystem>
  </xsl:template>

  <xsl:template match="user" mode="mailuser">
    <mailuser>
      <xsl:attribute name="name">
        <xsl:call-template name="mangle-sysuser-name">
          <xsl:with-param name="name" select="sysuser/@name"/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:choose>
        <xsl:when test="mail/mailgroup">
          <xsl:attribute name="mailgroup-enabled">true</xsl:attribute>
        </xsl:when>
        <xsl:otherwise>
          <xsl:attribute name="mailgroup-enabled">false</xsl:attribute>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:copy-of select="sysuser/password"/>
      <xsl:copy-of select="mail/mailbox"/>
      <xsl:copy-of select="mail/alias"/>

      <xsl:if test="mail/forward">
        <redirect><xsl:value-of select="mail/forward/text()"/></redirect>
      </xsl:if>
      <xsl:if test="mail/mailgroup">
        <xsl:for-each select="mail/mailgroup/forward">
          <mailgroup-member><xsl:value-of select="text()"/></mailgroup-member>
        </xsl:for-each>
      </xsl:if>
      <xsl:if test="mail/autoresponder">
		<autoresponders enabled="true">
		  <autoresponder name="vacation">
			<xsl:choose>
			  <xsl:when test="mail/autoresponder/@state = 'true'">
				<xsl:attribute name="status">on</xsl:attribute>
			  </xsl:when>
			  <xsl:otherwise>
				<xsl:attribute name="status">off</xsl:attribute>
			  </xsl:otherwise>
			</xsl:choose>
			<text><xsl:value-of select="mail/autoresponder/text()"/></text>
		  </autoresponder>
		</autoresponders>
      </xsl:if>

    </mailuser>
  </xsl:template>

  <xsl:template match="anonftp">
    <anonftp>
      <xsl:attribute name="pub">
        <xsl:choose>
          <xsl:when test="@cid">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="cid">
        <xsl:value-of select="@cid"/>
      </xsl:attribute>
      <xsl:attribute name="incoming">
        <xsl:choose>
          <xsl:when test="@cid_incoming">true</xsl:when>
          <xsl:otherwise>false</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
      <xsl:attribute name="cid_incoming">
        <xsl:value-of select="@cid_incoming"/>
      </xsl:attribute>

      <xsl:if test="limit[@name = 'disk-quota']">
        <anonftp-limit name="incoming-disk-quota"><xsl:value-of select="limit[@name = 'disk-quota']/text()"/></anonftp-limit>
      </xsl:if>
      <xsl:if test="limit[@name = 'account']">
        <anonftp-limit name="max-connections"><xsl:value-of select="limit[@name = 'account']/text()"/></anonftp-limit>
      </xsl:if>

      <anonftp-permission name="incoming-mkdir"/>
      <anonftp-permission name="incoming-download"/>

    </anonftp>
  </xsl:template>

  <xsl:template name="create-scripting">
    <scripting>
      <xsl:call-template name="convert-boolean-permission">
        <xsl:with-param name="raqname" select="'cgi'"/>
        <xsl:with-param name="pleskname" select="'cgi'"/>
      </xsl:call-template>

      <xsl:call-template name="convert-boolean-permission">
        <xsl:with-param name="raqname" select="'ssi'"/>
        <xsl:with-param name="pleskname" select="'ssi'"/>
      </xsl:call-template>

      <xsl:call-template name="convert-boolean-permission">
        <xsl:with-param name="raqname" select="'php'"/>
        <xsl:with-param name="pleskname" select="'php'"/>
      </xsl:call-template>

      <xsl:call-template name="convert-boolean-permission">
        <xsl:with-param name="raqname" select="'casp'"/>
        <xsl:with-param name="pleskname" select="'asp'"/>
      </xsl:call-template>

    </scripting>
  </xsl:template>

  <xsl:template name="mangle-sysuser-name">
    <xsl:param name="name"/>
    <xsl:choose>
      <xsl:when test="$name = 'admin' or $name = 'ftp' or $name = 'anonymous' or $name = 'root' or $name = 'mailman' or $name = 'tomcat4'"><xsl:value-of select="concat($name, '1')"/></xsl:when>
      <xsl:otherwise><xsl:value-of select="$name"/></xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template name="create-sysuser">
    <xsl:param name="user"/>

    <sysuser>
      <xsl:attribute name="name">
        <xsl:call-template name="mangle-sysuser-name">
          <xsl:with-param name="name" select="translate($user/sysuser/@name, '.', '_')"/>
        </xsl:call-template>
      </xsl:attribute>
      <xsl:if test="$user/permission[@name = 'shell']">
        <xsl:attribute name="shell">/bin/bash</xsl:attribute>
      </xsl:if>
      <xsl:copy-of select="$user/sysuser/password"/>
    </sysuser>
  </xsl:template>

  <xsl:template match="user" mode="webuser">
    <webuser name="{@fullname}">
      <!-- sysuser -->
      <xsl:call-template name="create-sysuser">
        <xsl:with-param name="user" select="."/>
      </xsl:call-template>
    </webuser>
  </xsl:template>

  <xsl:template name="convert-boolean-permission">
    <xsl:param name="raqname"/>
    <xsl:param name="pleskname"/>
    <xsl:choose>
      <xsl:when test="permission[@name = $raqname and @value = 'on']">
        <xsl:attribute name="{$pleskname}">true</xsl:attribute>
      </xsl:when>
      <xsl:otherwise>
        <xsl:attribute name="{$pleskname}">false</xsl:attribute>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- - - - - - - - - - - - - - - - - - - - - - -
       Outputs Plesk's limit (used for domains
       and clients accounts)
       Parameters:
       limit - limit name
       value - limit value
       - - - - - - - - - - - - - - - - - - - - - - - -->
  <xsl:template name="limit">
    <xsl:param name="limit"/>
    <xsl:param name="value"/>
    <xsl:element name="limit">
      <xsl:attribute name="name">
        <xsl:value-of select="$limit"/>
      </xsl:attribute>
      <xsl:value-of select="format-number($value, '#')"/>
    </xsl:element>
  </xsl:template>

</xsl:stylesheet>
