import logging
from collections import namedtuple
import xml.etree.ElementTree as et

from parallels.utils import ip

MoveList = namedtuple('MoveList', ('name', 'web_ip_v4', 'web_ip_v4_type', 'web_ip_v6', 'web_ip_v6_type'))

logger = logging.getLogger(__name__)

def read_move_list(fileobj):
	"""Methpd reads move list from file object.
	Example of move list XML:
	<move>
		<webspace name="test.tld">
			<web-ip type="shared">10.58.1.1</web-ip>
			<web-ip type="shared">::1</web-ip>
		</webspace>
	</move>
	:param fileobj: file object
	:return: MoveList
	"""

	webspace_name = None
	web_ip_v4 = None
	web_ip_v4_type = None
	web_ip_v6 = None
	web_ip_v6_type = None

	errors = []

	try:
		root_node = et.fromstring(fileobj.read())
	except Exception as err:
		logger.debug(u"Exception:", exc_info=True)
		errors.append("Failed to read move list file. Make sure it is a valid XML. Exception message: %s" % (str(err),))
	else:
		for webspace_xml_node in root_node:
			if webspace_xml_node.tag == 'webspace':
				webspace_name = webspace_xml_node.attrib.get('name')
				if webspace_name is not None:
					if webspace_name.strip() != '':
						for web_ip_xml_node in webspace_xml_node:
							if web_ip_xml_node.tag == 'web-ip':
								target_ip = web_ip_xml_node.text
								target_ip_type = web_ip_xml_node.attrib.get('type')
								if target_ip_type not in [u'shared', u'exclusive']:
									errors.append("Target IP '%s' has not valid IP type '%s' is not valid" % (target_ip, target_ip_type,))
								if target_ip is not None:
									if ip.is_ipv4(target_ip):
										web_ip_v4 = target_ip
										web_ip_v4_type = target_ip_type
									elif ip.is_ipv6(target_ip):
										web_ip_v6 = target_ip
										web_ip_v6_type = target_ip_type
									else:
										errors.append("Target web service node IP '%s' specified for domain '%s' is not valid" % (target_ip, webspace_name,))
								else:
									errors.append("Target web service node IP is not specified for domain '%s'" % (webspace_name,))
							else:
								errors.append("Unknown service to migrate: <%s>" % (web_ip_xml_node.tag))
					else:
						errors.append("Attribute 'name' of <webspace> node can not be empty")
				else:
					errors.append("Attribute 'name' is not set for <webspace> node, but must be set for all <domain> nodes in move list")
			else:
				errors.append('Unknown node <%s>: expected <webspace> node' % (webspace_xml_node.tag,))

	move_list = MoveList(name=webspace_name,
						 web_ip_v4=web_ip_v4, web_ip_v4_type=web_ip_v4_type,
						 web_ip_v6=web_ip_v6, web_ip_v6_type=web_ip_v6_type)
	return move_list, errors

