import logging
import subprocess
import sys
import os
from pipes import quote

from parallels.core import messages
from parallels.core.utils.common import safe_format, popen_no_inherit

logger = logging.getLogger(__name__)


class MigrationError(Exception):
    def __init__(self, message, error_id=None):
        """Class constructor

        :type message: str | unicode
        :type error_id: str | unicode
        """
        super(Exception, self).__init__(message)
        self._error_id = error_id

    @property
    def error_id(self):
        """Error ID - a string identifying the error for other components"""
        return self._error_id


class ParallelExecutionError(MigrationError):
    def __init__(self):
        super(MigrationError, self).__init__(messages.EXCEPTION_PARALLEL_EXECUTION)


class CommandExecutionError(MigrationError):
    """Exception occurred when executing some command on some host"""

    def __init__(self, message, stdout='', stderr='', host_description='', cause=''):
        super(MigrationError, self).__init__(message)
        self._stdout = stdout
        self._stderr = stderr
        self._host_description = host_description
        self._cause = cause

    @property
    def stdout(self):
        """stdout of the command"""
        return self._stdout

    @property
    def stderr(self):
        """stderr of the command"""
        return self._stderr

    @property
    def host_description(self):
        """Description of the host where the command was executed"""
        return self._host_description

    @property
    def cause(self):
        """Root cause of the error - exception caused command execution to fail"""
        return self._cause


class APIError(MigrationError):
    """Exception occurred when executing some request through external API"""

    @property
    def how_to_reproduce(self):
        """User-friendly description on how to reproduce API request"""
        raise NotImplementedError()


class MigrationConfigurationFileError(MigrationError):
    pass


class MigrationNoContextError(MigrationError):
    pass


class MigrationNoRepeatError(MigrationError):
    """Exception explicitly stating that there is no sense to repeat failed operation for the object

    For example, if subscription creation failed because such subscription already exists,
    there is no sense to try to create subscription any more times.
    """
    pass


class MigrationStopError(MigrationError):
    """Exception for situations when migration should be stopped, for example failed essential pre-migration checks"""
    pass


def local_command_exec(
    command, stdin_content=None, output_codepage='utf-8', error_policy='strict', env=None, working_dir=None
):
    if type(command) is list or type(command) is tuple:
        command_str = ' '.join([quote(c) for c in command])
        command_encoded = [c.encode('utf-8') for c in command]
    else:
        command_str = command
        command_encoded = command.encode('utf-8')

    command_env = os.environ.copy()
    command_env['LANG'] = 'en_US.utf-8'
    if env is not None:
        command_env.update(env)

    try:
        p = popen_no_inherit(
            command_encoded,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            env=command_env,
            cwd=working_dir
        )
        stdout_str, stderr_str = [s.decode(output_codepage, error_policy) for s in p.communicate(stdin_content)]
        return p.returncode, stdout_str, stderr_str
    except Exception as e:
        logger.debug(messages.LOG_EXCEPTION, exc_info=True)
        raise Exception(safe_format(
            messages.FAILED_TO_EXECUTE_LOCAL_COMMAND, command=command_str, reason=e
        )), None, sys.exc_info()[2]


def version_tuple(version):
    """Convert string product version to a tuple of ints"""
    return tuple(map(int, version.split('.')))
