import os
import shutil

from parallels.core import messages
from parallels.core.actions.base.common_action import CommonAction
from parallels.core.registry import Registry
from parallels.core.utils.common.logging import create_safe_logger
from parallels.core.utils.common_constants import LOCK_QUEUE
from parallels.core.utils.locks.session_file_lock import SessionFileLock
from parallels.core.utils.stop_mark import StopMark

logger = create_safe_logger(__name__)


class StopQueueTaskAction(CommonAction):
    """Stop and remove specified task from the queue"""

    def get_description(self):
        """Get short description of action as string

        :rtype: str
        """
        return messages.ACTION_STOP_QUEUE_TASK_DESCRIPTION

    def get_failure_message(self, global_context):
        """Get message for situation when action failed

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        return messages.ACTION_STOP_QUEUE_TASK_FAILURE

    def run(self, global_context):
        """Run action

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        """
        queue_lock = SessionFileLock(LOCK_QUEUE)
        queue_lock.acquire_block()

        task_id = global_context.options.task_id

        try:
            running_task_id = global_context.session_files.get_running_task_file().read()

            if task_id == running_task_id:
                StopMark.set()
                migration_lock = Registry.get_instance().get_migration_lock()
                migration_lock.acquire_block()
                try:
                    StopMark.remove()
                finally:
                    migration_lock.release()
            else:
                task_dir = os.path.join(global_context.session_files.get_queue_tasks_dir(), task_id)
                if os.path.isdir(task_dir):
                    shutil.move(task_dir, global_context.session_files.get_finished_tasks_dir())
        finally:
            queue_lock.release()
