import logging

from parallels.core import messages
from parallels.core.runners.base import BaseRunner
from parallels.core.runners.exceptions.base import BaseRunnerException
from parallels.core.utils import windows_utils
from parallels.core.utils.common import cached, safe_string_repr

logger = logging.getLogger(__name__)


class WindowsRunner(BaseRunner):
    """Add codepage detection to migrator's Windows CMD interface classes."""
    @property
    @cached
    def codepage(self):
        """Determine code page used by Windows CMD.

        'CHCP' utility returns Windows codepage, for example:
        "Active code page: 123"

        Python codec name is derived from 'chcp' output by adding 'cp' to code
        page number, for example: '866' -> 'cp866'

        Returns: Python codec name to be used for decoding CMD output.
        """

        code, stdout, stderr = self._run_command_and_decode_output(
                "cmd.exe /c CHCP", 'ascii', error_policy='ignore')
        if code != 0:
            raise BaseRunnerException(
                messages.UNABLE_DETECT_WINDOWS_CODEPAGE % (
                    safe_string_repr(stdout), safe_string_repr(stderr)
                )
            )
        else:
            codepage = u"cp%s" % stdout.split()[-1].strip('.')
            logger.debug(messages.DEBUG_DETECTED_CODEPAGE % codepage)

        return codepage

    def _run_unchecked_no_logging(
        self, cmd, args=None, stdin_content=None, output_codepage=None, error_policy='strict', env=None
    ):
        cmd_str = self._format_run_command(cmd, args)
        return self._sh_unchecked_no_logging(cmd_str, None, stdin_content, output_codepage, error_policy, env)

    def _run_command_and_decode_output(self, cmd_str, output_codepage=None, error_policy='strict', env=None):
        """Run a command, return its output decoded to UTF."""
        raise NotImplementedError()

    def file_exists(self, filename):
        raise NotImplementedError()

    def _format_run_command(self, cmd, args=None):
        """Format command arguments passed for 'run' and 'run_unchecked' methods into command line string

        This command line will be added to log and error messages, so it is expected that you could
        simply copy-paste it from logs or migrator output, and execute.

        :type cmd: str | unicode
        :type args: list[str | unicode] | None
        :rtype: str | unicode
        """
        return windows_utils.format_command_list(cmd, args)

    def _format_sh_command(self, cmd, args=None):
        """Format command arguments passed for 'sh' and 'sh_unchecked' methods into command line string

        This command line will be added to log and error messages, so it is expected that you could
        simply copy-paste it from logs or migrator output, and execute.

        :type cmd: str | unicode
        :type args: dict[str | unicode, str | unicode] | None
        """
        if args is not None and len(args) > 0:
            return windows_utils.format_command(cmd, **args)
        else:
            return cmd
