import logging
import threading

from parallels.core import messages
from parallels.core.utils.json_utils import encode_json, decode_json

logger = logging.getLogger(__name__)


class DataStorage(object):
    def __init__(self):
        self._data = {}
        try:
            self._data = self._load()
        except Exception:
            logger.debug(messages.LOG_EXCEPTION, exc_info=True)

    def get_data(self, key, default=None):
        """Retrieve data from storage

        :type key: str | None
        :type default: mixed | None
        :rtype: dict
        """
        return self._data.get(key, default)

    def set_data(self, key, data):
        """Place data into storage

        :type key: str
        :type data: mixed
        """
        self._data[key] = data
        try:
            self._save(self._data)
        except Exception:
            logger.debug(messages.LOG_EXCEPTION, exc_info=True)

    def _load(self):
        raise NotImplementedError()

    def _save(self, data):
        raise NotImplementedError()


class SessionFileDataStorage(DataStorage):
    def __init__(self, session_file):
        """
        :type session_file: parallels.core.session_files.SessionFile
        """
        self._session_file = session_file
        self._lock = threading.Lock()
        super(SessionFileDataStorage, self).__init__()

    def _load(self):
        with self._lock:
            return decode_json(self._session_file.read())

    def _save(self, data):
        with self._lock:
            self._session_file.write(encode_json(data))
