import logging

from parallels.core import messages
from parallels.core.actions.backup.create_converted import CreateConverted
from parallels.core.actions.backup.remove_content import RemoveContent
from parallels.core.actions.backup.save_converted import SaveConverted
from parallels.core.actions.base.compound_action import CompoundAction
from parallels.core.actions.base.entry_point_action import EntryPointAction
from parallels.core.actions.base.legacy_action import LegacyAction
from parallels.core.actions.check.check_target_account import CheckTargetAccountAction
from parallels.core.actions.check_remote_mssql import CheckRemoteMSSQL
from parallels.core.actions.check_remote_mssql_blocking import CheckRemoteMSSQLBlocking
from parallels.core.actions.check_upgrade_allowed import CheckUpgradeAllowedAction
from parallels.core.actions.close_ssh_connections import CloseSSHConnections
from parallels.core.actions.content.copy_content_after_transfer import CopyContentAfterTransfer
from parallels.core.actions.content.database.copy_database_content import CopyDatabaseContent
from parallels.core.actions.content.mail.copy_content import CopyMailContent
from parallels.core.actions.content.web.list_files import ListFiles
from parallels.core.actions.deploy.customers import DeployCustomers
from parallels.core.actions.deploy.database_users import DeployDatabaseUsers
from parallels.core.actions.deploy.databases import DeployDatabases
from parallels.core.actions.deploy.domain_aliases import DeployDomainAliases
from parallels.core.actions.deploy.domains import DeployDomains
from parallels.core.actions.deploy.extensions.not_supported import DeployExtensionsNotSupported
from parallels.core.actions.deploy.mail_accounts import DeployMailAccounts
from parallels.core.actions.deploy.panel_user_roles import DeployPanelUserRoles
from parallels.core.actions.deploy.panel_users import DeployPanelUsers
from parallels.core.actions.deploy.resellers import DeployResellers
from parallels.core.actions.deploy.resellers_print_report import DeployResellersPrintReport
from parallels.core.actions.deploy.subdomains import DeploySubdomains
from parallels.core.actions.deploy.subscription import DeploySubscription
from parallels.core.actions.dns.forwarding.set_dns_forwarding_not_supported \
    import SetDNSForwardingNotSupported
from parallels.core.actions.dns.forwarding.undo_dns_forwarding_not_supported \
    import UndoDNSForwardingNotSupported
from parallels.core.actions.dns.timings.set_dns_timings_not_supported \
    import SetDNSTimingsNotSupported
from parallels.core.actions.fetch_target import FetchTargetAction
from parallels.core.actions.final_report.compose_report import ComposeFinalMigrationReport
from parallels.core.actions.final_report.print_report import PrintFinalMigrationReport
from parallels.core.actions.fix_web_config_integrated_mode import FixWebConfigIntegratedMode
from parallels.core.actions.generate_html_log import GenerateHtmlLog
from parallels.core.actions.get_progress import GetProgressAction
from parallels.core.actions.hosting_analyser.analyse import AnalyseHosting
from parallels.core.actions.hosting_settings.check.check_database_conflicts import CheckDatabaseConflicts
from parallels.core.actions.hosting_settings.check.check_protected_dir_names import CheckProtectedDirNames
from parallels.core.actions.hosting_settings.check.check_service_plan_accordance import CheckServicePlanAccordance
from parallels.core.actions.hosting_settings.check.email_empty_password \
    import EmailEmptyPassword
from parallels.core.actions.hosting_settings.check.old_mysql_passwords import CheckOldMySQLPasswords
from parallels.core.actions.hosting_settings.convert.change_ftp_login import ChangeFTPLogin
from parallels.core.actions.hosting_settings.convert.change_hosting_subscription_web_ips import \
    ChangeHostingSubscriptionWebIPs
from parallels.core.actions.hosting_settings.convert.change_smartermail_password \
    import ChangeSmartermailPassword
from parallels.core.actions.hosting_settings.convert.change_subscription_document_root import \
    ChangeSubscriptionDocumentRootAction
from parallels.core.actions.hosting_settings.convert.change_subscription_mail_ips import ChangeSubscriptionMailIPs
from parallels.core.actions.hosting_settings.convert.change_sysuser_login import ChangeSysuserLogin
from parallels.core.actions.hosting_settings.convert.change_sysuser_password \
    import ChangeSysuserPassword
from parallels.core.actions.hosting_settings.convert.change_webmail_to_horde \
    import ChangeWebmailToHorde
from parallels.core.actions.hosting_settings.convert.fix_protected_dir_user_login import FixProtectedDirUserLogin
from parallels.core.actions.hosting_settings.convert.fix_windows_php53_isapi import FixWindowsPHP53ISAPI
from parallels.core.actions.hosting_settings.convert.no_password_mailuser_disable_cp_access import \
    NoPasswordMailuserDisableCPAccess
from parallels.core.actions.hosting_settings.convert.remap_databases \
    import RemapDatabases
from parallels.core.actions.hosting_settings.convert.remove_bad_protected_dirs import RemoveBadProtectedDirs
from parallels.core.actions.hosting_settings.convert.remove_subscription_default_db_server \
    import RemoveSubscriptionDefaultDBServer
from parallels.core.actions.hosting_settings.convert.remove_subscription_external_id \
    import RemoveSubscriptionExternalId
from parallels.core.actions.hosting_settings.convert.remove_subscription_limits_and_permissions \
    import RemoveSubscriptionLimitsAndPermissions
from parallels.core.actions.hosting_settings.convert.remove_subscription_to_plan_relation \
    import RemoveSubscriptionToPlanRelation
from parallels.core.actions.hosting_settings.convert_hosting_settings import ConvertHostingSettingsAction
from parallels.core.actions.hosting_settings.import_backups import ImportBackups
from parallels.core.actions.hosting_settings.remove_imported_backups import RemoveImportedBackups
from parallels.core.actions.hosting_settings.restore_subscription_hosting_settings import \
    RestoreSubscriptionHostingSettings
from parallels.core.actions.hosting_settings.restore_sysuser_logins \
    import RestoreSysuserLogins
from parallels.core.actions.hosting_settings.set_external_id_not_supported import SetExternalIdNotSupported
from parallels.core.actions.import_plans_not_supported import ImportPlansNotSupported
from parallels.core.actions.list_ip_addresses import ListIPAddresses
from parallels.core.actions.migration_list.check_migration_list_exists_for_migration \
    import CheckMigrationListExistsForMigration
from parallels.core.actions.migration_list.check_migration_list_not_exists_for_generation \
    import CheckMigrationListNotExistsForGeneration
from parallels.core.actions.migration_list.generate import GenerateMigrationList
from parallels.core.actions.post_migration_checks.print_report import PrintPostMigrationReportAction
from parallels.core.actions.post_migration_checks.print_subscription_report import \
    PrintSubscriptionPostMigrationReportAction
from parallels.core.actions.post_migration_checks.test_databases import TestDatabasesAction
from parallels.core.actions.post_migration_checks.test_dns import TestDnsAction
from parallels.core.actions.post_migration_checks.test_mail import TestMailAction
from parallels.core.actions.post_migration_checks.test_sites import TestSitesAction
from parallels.core.actions.post_migration_checks.test_users import TestUsersAction
from parallels.core.actions.pre_migration_checks.print_test_services_report import PrintTestServicesReportAction
from parallels.core.actions.pre_migration_checks.test_services import TestServicesAction
from parallels.core.actions.print_subscription_status import PrintSubscriptionStatus
from parallels.core.actions.queue.add_queue_task import AddQueueTaskAction
from parallels.core.actions.queue.remove_queue_subscriptions import RemoveQueueSubscriptionsAction
from parallels.core.actions.queue.run_queue import RunQueueAction
from parallels.core.actions.queue.stop_queue_task import StopQueueTaskAction
from parallels.core.actions.quick_check_cli_options import QuickCheckCLIOptions
from parallels.core.actions.read_ip_mapping import ReadIPMapping
from parallels.core.actions.remove_ssh_keys import RemoveSSHKeys
from parallels.core.actions.remove_subscriptions_from_target import RemoveSubscriptionsFromTargetAction
from parallels.core.actions.run_subscription_migration_finished_scripts import RunSubscriptionMigrationFinishedScripts
from parallels.core.actions.shutdown_windows_rsync_servers import ShutdownWindowsRsyncServers
from parallels.core.actions.statistics.finalize_session import FinalizeSessionStatistics
from parallels.core.actions.statistics.start_session import StartSessionStatistics
from parallels.core.actions.stop_migration import StopMigrationAction
from parallels.core.actions.transfer_agent.start import Start as StartTransferAgent
from parallels.core.actions.transfer_agent.stop import Stop as StopTransferAgent
from parallels.core.actions.uninstall_dump_agent import UninstallDumpAgentAction
from parallels.core.actions.utils.logging_properties import LoggingProperties
from parallels.core.actions.windows_security_policy.restore_policy import RestoreWindowsSecurityPolicy
from parallels.core.actions.windows_security_policy.set_policy import SetWindowsSecurityPolicy
from parallels.core.reports.printer import print_report
from parallels.core.utils.backup_adapter import SubscriptionBackup
from parallels.core.utils.backup_adapter import SubscriptionBackupRaw
from parallels.core.workflow.extension import WorkflowExtensionBase
from parallels.plesk.actions.deploy.hosting_plans import DeployHostingPlans
from parallels.plesk.actions.deploy.reseller_plans import DeployResellerPlans
from parallels.plesk.source.plesk.actions.content.web.check_target_web_hosting \
    import CheckTargetWebHosting


logger = logging.getLogger(__name__)


class SharedHostingWorkflowExtension(WorkflowExtensionBase):
    """Common workflow for shared hosting migration"""

    def extend_workflow(self, workflow):
        self._configure_shared_actions(workflow)
        self._configure_entry_points(workflow)

    def _configure_shared_actions(self, workflow):
        self._configure_shared_action_fetch_source(workflow)
        self._configure_shared_action_fetch_source_shallow(workflow)
        self._configure_shared_action_fetch_target(workflow)
        self._configure_shared_action_check_infrastructure(workflow)
        self._configure_shared_action_sync_web_content_assets(workflow)
        self._configure_shared_action_copy_web_files(workflow)
        self._configure_shared_action_copy_web_content(workflow)
        self._configure_shared_action_copy_mail_content(workflow)
        self._configure_shared_action_analyse_hosting(workflow)
        self._configure_shared_action_cleanup(workflow)
        self._configure_shared_action_copy_db_content(workflow)
        self._configure_shared_action_copy_content(workflow)
        self._configure_shared_action_check_connections(workflow)
        self._configure_shared_action_check_target_account(workflow)
        self._configure_shared_action_check_sources(workflow)
        self._configure_shared_action_check_target(workflow)
        self._configure_shared_action_check_target_licenses(workflow)
        self._configure_shared_action_read_migration_list(workflow)
        self._configure_shared_action_read_ip_mapping(workflow)
        self._configure_shared_action_convert(workflow)
        self._configure_shared_action_verify_hosting(workflow)
        self._configure_shared_action_convert_hosting(workflow)
        self._configure_shared_action_initial_pre_checks(workflow)
        self._configure_shared_action_prepare_assets(workflow)
        self._configure_shared_action_post_migration_finalize(workflow)

    def _configure_entry_points(self, workflow):
        self._configure_entry_point_get_progress(workflow)
        self._configure_entry_point_check_upgrade_allowed(workflow)
        self._configure_entry_point_run_queue(workflow)
        self._configure_entry_point_add_queue_task(workflow)
        self._configure_entry_point_stop_queue_task(workflow)
        self._configure_entry_point_stop_migration(workflow)
        self._configure_entry_point_start_transfer_agent(workflow)
        self._configure_entry_point_analyse_hosting(workflow)
        self._configure_entry_point_generate_migration_list(workflow)
        self._configure_entry_point_check(workflow)
        self._configure_entry_point_check_infrastructure(workflow)
        self._configure_entry_point_check_target_licenses(workflow)
        self._configure_entry_point_check_target_account(workflow)
        self._configure_entry_point_deploy_resellers(workflow)
        self._configure_entry_point_deploy_hosting_plans(workflow)
        self._configure_entry_point_deploy_extensions(workflow)
        self._configure_entry_point_transfer_accounts(workflow)
        self._configure_entry_point_copy_content(workflow)
        self._configure_entry_point_copy_mail_content(workflow)
        self._configure_entry_point_copy_web_content(workflow)
        self._configure_entry_point_copy_db_content(workflow)
        self._configure_entry_point_verify_hosting(workflow)
        self._configure_entry_point_set_dns_forwarding(workflow)
        self._configure_entry_point_undo_dns_forwarding(workflow)
        self._configure_entry_point_set_low_dns_timings(workflow)
        self._configure_entry_point_set_external_id(workflow)
        self._configure_entry_point_test_all(workflow)
        self._configure_entry_point_test_sites(workflow)
        self._configure_entry_point_test_dns(workflow)
        self._configure_entry_point_test_mail(workflow)
        self._configure_entry_point_test_users(workflow)
        self._configure_entry_point_test_databases(workflow)
        self._configure_entry_point_test_services(workflow)
        self._configure_entry_point_list_ip_addresses(workflow)
        self._configure_entry_point_list_files_to_copy(workflow)
        self._configure_entry_point_generate_html_log(workflow)
        self._configure_entry_point_remove_subscriptions_from_target(workflow)
        self._configure_entry_point_remove_subscriptions_from_queue(workflow)

    @staticmethod
    def _configure_shared_action_fetch_source(workflow):
        workflow.add_shared_action('fetch-source', CompoundAction(
            description=messages.FETCH_INFORMATION_FROM_SOURCE_PANEL))

    @staticmethod
    def _configure_shared_action_fetch_source_shallow(workflow):
        # By default, consider shallow backup is the same as full backup,
        # and fetch source shallow is the same as fetch source
        workflow.add_shared_action(
            'fetch-source-shallow',
            workflow.get_shared_action('fetch-source')
        )

    @staticmethod
    def _configure_shared_action_fetch_target(workflow):
        workflow.add_shared_action('fetch-target', FetchTargetAction())

    @staticmethod
    def _configure_shared_action_check_infrastructure(workflow):
        workflow.add_shared_action('check-infrastructure', LegacyAction(
            description=messages.CHECK_INFRASTRUCTURE_CONNECTIONS_AND_DISK_SPACE,
            function=lambda ctx: ctx.migrator.check_infrastructure(ctx.options)
        ))

    @staticmethod
    def _configure_shared_action_copy_content(workflow):
        workflow.add_shared_action('copy-content', CompoundAction(
            description=messages.COPY_WEB_DATABASE_AND_MAIL_CONTENT,
            run_by_subscription=True
        ))
        workflow.get_shared_action('copy-content').insert_action(
            'web', workflow.get_shared_action_pointer('copy-web-content')
        )
        workflow.get_shared_action('copy-content').insert_action(
            'database', workflow.get_shared_action_pointer('copy-db-content')
        )
        workflow.get_shared_action('copy-content').insert_action(
            'mail', workflow.get_shared_action_pointer('copy-mail-content')
        )

    @staticmethod
    def _configure_shared_action_copy_web_files(workflow):
        workflow.add_shared_action('copy-web-files', CompoundAction(
            description=messages.ACTION_COPY_WEB_FILES
        ))

    @staticmethod
    def _configure_shared_action_copy_web_content(workflow):
        workflow.add_shared_action('copy-web-content', CompoundAction(
            description=messages.COPY_WEB_FILES_AND_SYNC_WEB))
        workflow.get_shared_action('copy-web-content').insert_action(
            'copy-web-files', workflow.get_shared_action('copy-web-files')
        )
        workflow.get_shared_action('copy-web-content').insert_action(
            'web-assets', workflow.get_shared_action('sync-web-content-assets')
        )

    @staticmethod
    def _configure_shared_action_sync_web_content_assets(workflow):
        workflow.add_shared_action('sync-web-content-assets', CompoundAction(
            description=messages.ACTION_SYNC_WEB_CONTENT_ASSETS
        ))
        workflow.get_shared_action('sync-web-content-assets').insert_action(
            'fix-web-config-classic-mode', FixWebConfigIntegratedMode()
        )

    @staticmethod
    def _configure_shared_action_copy_mail_content(workflow):
        workflow.add_shared_action('copy-mail-content', CopyMailContent())

    @staticmethod
    def _configure_shared_action_analyse_hosting(workflow):
        workflow.add_shared_action('analyse-hosting', AnalyseHosting())

    @staticmethod
    def _configure_shared_action_cleanup(workflow):
        workflow.add_shared_action('cleanup', CompoundAction(
            description='Perform final clean up',
            logging_properties=LoggingProperties(compound=False)
        ))
        workflow.get_shared_action('cleanup').insert_action(
            'uninstall-agent', UninstallDumpAgentAction()
        )
        workflow.get_shared_action('cleanup').insert_action(
            'remove-ssh-keys', RemoveSSHKeys()
        )
        workflow.get_shared_action('cleanup').insert_action(
            'close-ssh-connections', CloseSSHConnections()
        )
        workflow.get_shared_action('cleanup').insert_action(
            'shutdown-windows-rsync-servers', ShutdownWindowsRsyncServers()
        )
        workflow.get_shared_action('cleanup').insert_action(
            'shutdown-windows-agents', StopTransferAgent()
        )

    @staticmethod
    def _configure_shared_action_copy_db_content(workflow):
        workflow.add_shared_action('copy-db-content', CopyDatabaseContent())

    @staticmethod
    def _configure_shared_action_check_connections(workflow):
        workflow.add_shared_action('check-connections', LegacyAction(
            description=messages.ACTION_CHECK_CONNECTIONS,
            function=lambda ctx: ctx.migrator._check_connections(ctx.options)
        ))

    @staticmethod
    def _configure_shared_action_check_sources(workflow):
        workflow.add_shared_action('check-sources', LegacyAction(
            description=messages.ACTION_CHECK_SOURCE_CONNECTIONS,
            function=lambda ctx: ctx.migrator._check_sources()
        ))

    @staticmethod
    def _configure_shared_action_check_target(workflow):
        workflow.add_shared_action('check-target', LegacyAction(
            description=messages.ACTION_CHECK_TARGET_CONNECTIONS,
            function=lambda ctx: ctx.migrator._check_target()
        ))

    @staticmethod
    def _configure_shared_action_check_target_licenses(workflow):
        workflow.add_shared_action('check-target-licenses', CompoundAction())

    @staticmethod
    def _configure_shared_action_check_target_account(workflow):
        workflow.add_shared_action('check-target-account', CheckTargetAccountAction())

    @staticmethod
    def _configure_shared_action_read_migration_list(workflow):
        workflow.add_shared_action('read-migration-list', LegacyAction(
            description=messages.ACTION_READ_MIGRATION_LIST,
            function=lambda ctx: ctx.migrator.read_migration_list_lazy()
        ))

    @staticmethod
    def _configure_shared_action_read_ip_mapping(workflow):
        workflow.add_shared_action('read-ip-mapping', ReadIPMapping())

    @staticmethod
    def _configure_shared_action_convert(workflow):
        workflow.add_shared_action('convert', LegacyAction(
            description=messages.CONVERT_CLIENTS_AND_SUBSCRIPTIONS,
            function=lambda ctx: ctx.migrator.convert()
        ))

    @staticmethod
    def _configure_shared_action_verify_hosting(workflow):
        workflow.add_shared_action('verify-hosting', CompoundAction(
            description=messages.ACTION_VERIFY_HOSTING
        ))

    @staticmethod
    def _configure_shared_action_prepare_assets(workflow):
        workflow.add_shared_action('prepare-assets', CompoundAction())
        workflow.get_shared_action('prepare-assets').insert_action(
            'check-migration-list', CheckMigrationListExistsForMigration()
        )
        workflow.get_shared_action('prepare-assets').insert_action(
            'initial-pre-checks', workflow.get_shared_action('initial-pre-checks')
        )
        workflow.get_shared_action('prepare-assets').insert_action(
            'fetch-source-shallow', workflow.get_shared_action_pointer('fetch-source-shallow')
        )
        workflow.get_shared_action('prepare-assets').insert_action(
            'read-migration-list', workflow.get_shared_action('read-migration-list')
        )
        workflow.get_shared_action('prepare-assets').insert_action(
            'read-ip-mapping', workflow.get_shared_action_pointer('read-ip-mapping')
        )
        workflow.get_shared_action('prepare-assets').insert_action(
            'fetch-source', workflow.get_shared_action('fetch-source')
        )
        workflow.get_shared_action('prepare-assets').insert_action(
            'fetch-target', workflow.get_shared_action_pointer('fetch-target')
        )
        workflow.get_shared_action('prepare-assets').insert_action(
            'convert-target-model', workflow.get_shared_action_pointer('convert')
        )
        workflow.get_shared_action('prepare-assets').insert_action(
            'convert-hosting', workflow.get_shared_action_pointer('convert-hosting')
        )

    @staticmethod
    def _configure_shared_action_post_migration_finalize(workflow):
        workflow.add_shared_action('post-migration-finalize', CompoundAction())
        workflow.get_shared_action('post-migration-finalize').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_shared_action('post-migration-finalize').insert_action(
            'compose-final-report', ComposeFinalMigrationReport()
        )
        workflow.get_shared_action('post-migration-finalize').insert_action(
            'print-final-report', PrintFinalMigrationReport()
        )

    @staticmethod
    def _configure_shared_action_initial_pre_checks(workflow):
        workflow.add_shared_action('initial-pre-checks', CompoundAction())
        workflow.get_shared_action('initial-pre-checks').insert_action(
            'check-cli-options', QuickCheckCLIOptions()
        )
        workflow.get_shared_action('initial-pre-checks').insert_action(
            'check-connections',
            workflow.get_shared_action_pointer('check-connections')
        )
        workflow.get_shared_action('initial-pre-checks').insert_action(
            'check-services', TestServicesAction()
        )
        workflow.get_shared_action('initial-pre-checks').insert_action(
            'print-services-report', PrintTestServicesReportAction()
        )
        workflow.get_shared_action('initial-pre-checks').insert_action(
            'check-target-licenses',
            workflow.get_shared_action('check-target-licenses')
        )

    @staticmethod
    def _configure_entry_point_generate_migration_list(workflow):
        workflow.add_entry_point('generate-migration-list', EntryPointAction())

        workflow.get_path('generate-migration-list').insert_action(
            'check-migration-list-exists', CheckMigrationListNotExistsForGeneration()
        )
        workflow.get_path('generate-migration-list').insert_action(
            'initial-pre-checks', workflow.get_shared_action('initial-pre-checks')
        )
        workflow.get_path('generate-migration-list').insert_action(
            'fetch-source', workflow.get_shared_action_pointer('fetch-source-shallow')
        )
        workflow.get_path('generate-migration-list').insert_action(
            'generate-migration-list', GenerateMigrationList(),
        )
        workflow.get_path('generate-migration-list').insert_action(
            'analyse-hosting', workflow.get_shared_action('analyse-hosting')
        )

        workflow.get_path('generate-migration-list').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('generate-migration-list').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_check(workflow):
        workflow.add_entry_point('check', EntryPointAction())
        workflow.get_path('check').insert_action(
            'prepare-assets', workflow.get_shared_action('prepare-assets')
        )
        workflow.get_path('check').insert_action('check-data', CompoundAction())
        workflow.get_path('check/check-data').insert_action(
            'check-email-empty-password', EmailEmptyPassword(SubscriptionBackupRaw())
        )
        workflow.get_path('check/check-data').insert_action('check-email-empty-password', CheckOldMySQLPasswords())
        workflow.get_path('check/check-data').insert_action(
            'check-protected-dir-name', CheckProtectedDirNames(SubscriptionBackupRaw())
        )
        workflow.get_path('check').insert_action('check-remote-mssql', CheckRemoteMSSQL())
        workflow.get_path('check').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('check').insert_action(
            'print-report', LegacyAction(
                description=messages.ACTION_PRINT_PRE_MIGRATION_ERROR,
                function=lambda ctx: print_report(
                    ctx.pre_check_report, "pre_migration_report_tree",
                    show_no_issue_branches=False
                ),
                is_critical=False
            )
        )
        workflow.get_path('check').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_check_infrastructure(workflow):
        workflow.add_entry_point('check-infrastructure', EntryPointAction())
        workflow.get_path('check-infrastructure').insert_action(
            'prepare-assets', workflow.get_shared_action('prepare-assets')
        )
        workflow.get_path('check-infrastructure').insert_action(
            'check-infrastructure', workflow.get_shared_action_pointer('check-infrastructure')
        )

    @staticmethod
    def _configure_entry_point_check_target_licenses(workflow):
        workflow.add_entry_point('check-target-licenses', EntryPointAction())
        workflow.get_path('check-target-licenses').insert_action(
            'check-target-licenses',
            workflow.get_shared_action_pointer('check-target-licenses')
        )

    @staticmethod
    def _configure_entry_point_check_target_account(workflow):
        workflow.add_entry_point('check-target-account', EntryPointAction())
        workflow.get_path('check-target-account').insert_action(
            'check-target-account',
            workflow.get_shared_action('check-target-account')
        )

    @staticmethod
    def _configure_entry_point_deploy_resellers(workflow):
        workflow.add_entry_point('deploy-resellers', EntryPointAction())
        workflow.get_path('deploy-resellers').insert_action(
            'prepare-assets', workflow.get_shared_action('prepare-assets')
        )
        workflow.get_path('deploy-resellers').insert_action('deploy-resellers', DeployResellers())
        workflow.get_path('deploy-resellers').insert_action('cleanup', workflow.get_shared_action('cleanup'))
        workflow.get_path('deploy-resellers').insert_action('print-report', DeployResellersPrintReport())
        workflow.get_path('deploy-resellers').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_deploy_hosting_plans(workflow):
        workflow.add_entry_point('deploy-hosting-plans', EntryPointAction())
        workflow.get_path('deploy-hosting-plans').insert_action(
            'check-target-panel', ImportPlansNotSupported()
        )

    @staticmethod
    def _configure_entry_point_deploy_extensions(workflow):
        workflow.add_entry_point('deploy-extensions', EntryPointAction())
        workflow.get_path('deploy-extensions').insert_action(
            'check-target-panel', DeployExtensionsNotSupported()
        )

    @staticmethod
    def _configure_shared_action_convert_hosting(workflow):
        workflow.add_shared_action('convert-hosting', ConvertHostingSettingsAction())
        convert_hosting = workflow.get_shared_action('convert-hosting')
        convert_hosting.insert_actions([
            ('create-converted-backup', CreateConverted(SubscriptionBackup())),
            ('remap-databases', RemapDatabases(SubscriptionBackup())),
            ('remove-subscription-to-plan-relation', RemoveSubscriptionToPlanRelation(SubscriptionBackup())),
            ('remove-subscription-default-db-server', RemoveSubscriptionDefaultDBServer(SubscriptionBackup())),
            ('remove-subscription-limits-and-permissions',
                RemoveSubscriptionLimitsAndPermissions(SubscriptionBackup())),
            ('change-webmail-to-horde', ChangeWebmailToHorde(SubscriptionBackup())),
            (
                'change-hosting-subscription-web-ips',
                ChangeHostingSubscriptionWebIPs(SubscriptionBackup())
            ),
            (
                'change-subscription-mail-ips',
                ChangeSubscriptionMailIPs(SubscriptionBackup())
            ),
            ('remove-subscription-external-id', RemoveSubscriptionExternalId(SubscriptionBackup())),
            ('change-ftp-login', ChangeFTPLogin(SubscriptionBackup())),
            ('change-sysuser-password', ChangeSysuserPassword(SubscriptionBackup())),
            ('change-smartermail-password', ChangeSmartermailPassword(SubscriptionBackup())),
            ('fix-windows-php53-isapi', FixWindowsPHP53ISAPI(SubscriptionBackup())),
            ('fix-protected-dir-logins', FixProtectedDirUserLogin(SubscriptionBackup())),
            ('remove-bad-protected-dirs', RemoveBadProtectedDirs(SubscriptionBackup())),
            (
                'disable-cp-access-for-mailuser-with-no-password',
                NoPasswordMailuserDisableCPAccess(SubscriptionBackup())
            ),
            ('check-database-conflicts', CheckDatabaseConflicts(SubscriptionBackup())),
            ('check-service-plan-accordance', CheckServicePlanAccordance(SubscriptionBackup())),
            ('backup-remove-content', RemoveContent(SubscriptionBackup())),
            ('check-email-empty-password', EmailEmptyPassword(SubscriptionBackup())),
            ('change-subscription-document-root', ChangeSubscriptionDocumentRootAction(SubscriptionBackup())),
            ('save-backup', SaveConverted(SubscriptionBackup())),
        ])

    @staticmethod
    def _configure_entry_point_transfer_accounts(workflow):
        workflow.add_entry_point('transfer-accounts', EntryPointAction())
        workflow.get_path('transfer-accounts').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('transfer-accounts').insert_action('check-main-node-disk-space-requirements', LegacyAction(
            description=messages.CHECK_DISK_SPACE_REQUIREMENTS_FOR_MAIN,
            function=lambda ctx: ctx.migrator.check_main_node_disk_space_requirements(ctx.options)
        ))
        workflow.get_path('transfer-accounts').insert_action(
            'check-remote-mssql-servers', CheckRemoteMSSQLBlocking()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'statistics-start-session', StartSessionStatistics()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'windows-refresh-components', LegacyAction(
                description=messages.REFRESH_COMPONENTS_LIST_TARGET_WINDOWS_SERVERS,
                function=lambda ctx: ctx.migrator._refresh_service_node_components_for_windows(),
                logging_properties=LoggingProperties(compound=False),
                is_critical=False,
            )
        )
        workflow.get_path('transfer-accounts').insert_action(
            'set-windows-security-policy', SetWindowsSecurityPolicy()
        )

        workflow.get_path('transfer-accounts').insert_action(
            'deploy', CompoundAction("Deploy hosting")
        )
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-reseller-plans', DeployResellerPlans())
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-resellers', DeployResellers())
        workflow.get_path('transfer-accounts/deploy').insert_action(
            'deploy-resellers-print-report', DeployResellersPrintReport()
        )
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-plans', DeployHostingPlans())
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-customers', DeployCustomers())
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-panel-user-roles', DeployPanelUserRoles())
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-panel-users', DeployPanelUsers())
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-subscription', DeploySubscription())

        workflow.get_path('transfer-accounts/deploy').insert_action(
            'restore-sysuser-logins', CompoundAction(
                description=messages.RESTORE_LOGINS_SYSTEM_USERS,
                run_by_subscription=False
            )
        )
        workflow.get_path('transfer-accounts/deploy/restore-sysuser-logins').insert_action(
            'change-sysuser-logins', RestoreSysuserLogins()
        )
        workflow.get_path('transfer-accounts/deploy/restore-sysuser-logins').insert_action(
            'change-backup', ChangeSysuserLogin(SubscriptionBackup())
        )
        workflow.get_path('transfer-accounts/deploy/restore-sysuser-logins').insert_action(
            'save-backup', SaveConverted(SubscriptionBackup()),
        )
        workflow.get_path('transfer-accounts/deploy').insert_action(
            'import-backups', ImportBackups()
        )

        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-domains', DeployDomains())
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-subdomains', DeploySubdomains())
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-domain-aliases', DeployDomainAliases())
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-mail-users', DeployMailAccounts())
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-databases', DeployDatabases())
        workflow.get_path('transfer-accounts/deploy').insert_action('deploy-database-users', DeployDatabaseUsers())

        workflow.get_path('transfer-accounts').insert_action(
            'restore-hosting', CompoundAction("Restore hosting settings")
        )
        workflow.get_path('transfer-accounts/restore-hosting').insert_action(
            'restore-hosting-settings', RestoreSubscriptionHostingSettings()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'verify-hosting', workflow.get_shared_action_pointer('verify-hosting')
        )

        workflow.get_path('transfer-accounts').insert_action(
            'copy-content', workflow.get_shared_action_pointer('copy-content')
        )
        workflow.get_shared_action('sync-web-content-assets').insert_action(
            'check-target-web-hosting', CheckTargetWebHosting()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'run-subscription-migration-finished-scripts', RunSubscriptionMigrationFinishedScripts()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'print-subscription-report', PrintSubscriptionStatus()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'restore-status', LegacyAction(
                description=messages.RESTORE_STATUS_SUSPENDED_CLIENTS_AND_SUBSCRIPTIONS,
                function=lambda ctx: ctx.migrator.restore_status(ctx.options),
                logging_properties=LoggingProperties(compound=False),
                is_critical=False
            )
        )
        workflow.get_path('transfer-accounts').insert_action(
            'remove-imported-backups', RemoveImportedBackups()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'restore-windows-security-policy', RestoreWindowsSecurityPolicy()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'restart-iis-for-idn-domains', LegacyAction(
                description=messages.ACTION_RESTART_IIS_FOR_IDN_DOMAINS,
                function=lambda ctx: ctx.migrator._fix_iis_idn_401_error(),
                logging_properties=LoggingProperties(compound=False),
                is_critical=False
            )
        )
        workflow.get_path('transfer-accounts').insert_action(
            'copy-content-after-transfer', CopyContentAfterTransfer()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'test-all-after-transfer', CompoundAction()
        )
        workflow.get_path('transfer-accounts/test-all-after-transfer').insert_action(
            'test-sites', TestSitesAction()
        )
        workflow.get_path('transfer-accounts/test-all-after-transfer').insert_action(
            'test-dns', TestDnsAction()
        )
        workflow.get_path('transfer-accounts/test-all-after-transfer').insert_action(
            'test-mail', TestMailAction()
        )
        workflow.get_path('transfer-accounts/test-all-after-transfer').insert_action(
            'test-users', TestUsersAction()
        )
        workflow.get_path('transfer-accounts/test-all-after-transfer').insert_action(
            'test-databases', TestDatabasesAction()
        )
        workflow.get_path('transfer-accounts/test-all-after-transfer').insert_action(
            'print-subscription-post-migration-report', PrintSubscriptionPostMigrationReportAction()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'print-post-migration-report', PrintPostMigrationReportAction('test_all_report')
        )
        workflow.get_path('transfer-accounts').insert_action(
            'compose-final-report', ComposeFinalMigrationReport()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'statistics-finalize-session', FinalizeSessionStatistics()
        )
        workflow.get_path('transfer-accounts').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('transfer-accounts').insert_action(
            'print-final-report', PrintFinalMigrationReport()
        )
        workflow.get_path('transfer-accounts').register_shutdown(
            'set-windows-security-policy',
            'restore-windows-security-policy'
        )
        workflow.get_path('transfer-accounts').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_copy_content(workflow):
        workflow.add_entry_point('copy-content', EntryPointAction())
        workflow.get_path('copy-content').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('copy-content').insert_action(
            'copy-content',
            workflow.get_shared_action_pointer('copy-content')
        )
        workflow.get_path('copy-content').insert_action(
            'print-subscription-report', PrintSubscriptionStatus()
        )
        workflow.get_path('copy-content').insert_actions(
            workflow.get_shared_action('post-migration-finalize').get_all_actions()
        )
        workflow.get_path('copy-content').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_copy_mail_content(workflow):
        workflow.add_entry_point('copy-mail-content', EntryPointAction())
        workflow.get_path('copy-mail-content').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('copy-mail-content').insert_action(
            'copy-mail-content',
            workflow.get_shared_action_pointer('copy-mail-content')
        )
        workflow.get_path('copy-mail-content').insert_actions(
            workflow.get_shared_action('post-migration-finalize').get_all_actions()
        )
        workflow.get_path('copy-mail-content').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_copy_web_content(workflow):
        workflow.add_entry_point('copy-web-content', EntryPointAction())
        workflow.get_path('copy-web-content').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('copy-web-content').insert_action(
            'copy-web-content',
            workflow.get_shared_action_pointer('copy-web-content')
        )
        workflow.get_path('copy-web-content').insert_actions(
            workflow.get_shared_action('post-migration-finalize').get_all_actions()
        )
        workflow.get_path('copy-web-content').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_copy_db_content(workflow):
        workflow.add_entry_point('copy-db-content', EntryPointAction())
        workflow.get_path('copy-db-content').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('copy-db-content').insert_action(
            'check-remote-mssql-servers', CheckRemoteMSSQLBlocking()
        )
        workflow.get_path('copy-db-content').insert_action(
            'copy-db-content',
            workflow.get_shared_action_pointer('copy-db-content')
        )
        workflow.get_path('copy-db-content').insert_actions(
            workflow.get_shared_action('post-migration-finalize').get_all_actions()
        )
        workflow.get_path('copy-db-content').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_verify_hosting(workflow):
        workflow.add_entry_point('verify-hosting', EntryPointAction())
        workflow.get_path('verify-hosting').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('verify-hosting').insert_action(
            'verify-hosting',
            workflow.get_shared_action_pointer('verify-hosting')
        )
        workflow.get_path('verify-hosting').insert_actions(
            workflow.get_shared_action('post-migration-finalize').get_all_actions()
        )
        workflow.get_path('verify-hosting').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_get_progress(workflow):
        workflow.add_entry_point('get-progress', EntryPointAction())
        workflow.get_path('get-progress').insert_action('get-progress', GetProgressAction())

    @staticmethod
    def _configure_entry_point_check_upgrade_allowed(workflow):
        workflow.add_entry_point('check-upgrade-allowed', EntryPointAction())
        workflow.get_path('check-upgrade-allowed').insert_action('check-upgrade-allowed', CheckUpgradeAllowedAction())

    @staticmethod
    def _configure_entry_point_run_queue(workflow):
        workflow.add_entry_point('run-queue', EntryPointAction())
        workflow.get_path('run-queue').insert_action('run-queue', RunQueueAction())

    @staticmethod
    def _configure_entry_point_add_queue_task(workflow):
        workflow.add_entry_point('add-queue-task', EntryPointAction())
        workflow.get_path('add-queue-task').insert_action('add-queue-task', AddQueueTaskAction())

    @staticmethod
    def _configure_entry_point_stop_migration(workflow):
        workflow.add_entry_point('stop-migration', EntryPointAction())
        workflow.get_path('stop-migration').insert_action('stop-migration', StopMigrationAction())

    @staticmethod
    def _configure_entry_point_stop_queue_task(workflow):
        workflow.add_entry_point('stop-queue-task', EntryPointAction())
        workflow.get_path('stop-queue-task').insert_action('stop-queue-task', StopQueueTaskAction())

    @staticmethod
    def _configure_entry_point_start_transfer_agent(workflow):
        workflow.add_entry_point('start-transfer-agent', EntryPointAction())
        workflow.get_path('start-transfer-agent').insert_action('start-transfer-agent', StartTransferAgent())

    @staticmethod
    def _configure_entry_point_analyse_hosting(workflow):
        workflow.add_entry_point('analyse-hosting', EntryPointAction())
        workflow.get_path('analyse-hosting').insert_action(
            'analyse-hosting',
            workflow.get_shared_action_pointer('analyse-hosting')
        )

    @staticmethod
    def _configure_entry_point_set_dns_forwarding(workflow):
        # Set up the DNS forwarding.
        #
        # Classic DNS forwarding just does not work by its design: source Plesk
        # server is authoritative for its zones - according to registrar - and
        # will not forward queries about these zones anywhere.  So the DNS
        # forwarding we do by turning source Plesk into slave mode for the
        # migrated zones.
        #
        # This step is separate from transfer-accounts because the
        # steps are following:
        # - transfer-accounts
        # - disable access to both source and target servers for the migrated
        # subscriptions MANUALLY
        # - set up DNS forwarding
        workflow.add_entry_point('set-dns-forwarding', EntryPointAction())
        workflow.get_path('set-dns-forwarding').insert_action(
            'check-source-not-supported', SetDNSForwardingNotSupported()
        )

    @staticmethod
    def _configure_entry_point_undo_dns_forwarding(workflow):
        workflow.add_entry_point('undo-dns-forwarding', EntryPointAction())
        # By default we don't have DNS forwarding support.
        # Should be overridden in child workflow for panels which support DNS forwarding.
        workflow.get_path('undo-dns-forwarding').insert_action(
            'check-source-not-supported', UndoDNSForwardingNotSupported()
        )

    @staticmethod
    def _configure_entry_point_set_low_dns_timings(workflow):
        # Set up low DNS timings
        workflow.add_entry_point('set-low-dns-timings', EntryPointAction())
        workflow.get_path('set-low-dns-timings').insert_action(
            'check-source-not-supported', SetDNSTimingsNotSupported()
        )

    @staticmethod
    def _configure_entry_point_set_external_id(workflow):
        workflow.add_entry_point('set-external-id', EntryPointAction())
        workflow.get_path('set-external-id').insert_action(
            'check-source-not-supported', SetExternalIdNotSupported()
        )

    @staticmethod
    def _configure_entry_point_test_all(workflow):
        workflow.add_entry_point('test-all', EntryPointAction())
        workflow.get_path('test-all').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('test-all').insert_action(
            'test-sites', TestSitesAction()
        )
        workflow.get_path('test-all').insert_action(
            'test-dns', TestDnsAction()
        )
        workflow.get_path('test-all').insert_action(
            'test-mail', TestMailAction()
        )
        workflow.get_path('test-all').insert_action(
            'test-users', TestUsersAction()
        )
        workflow.get_path('test-all').insert_action(
            'test-databases', TestDatabasesAction()
        )
        workflow.get_path('test-all').insert_action(
            'print-subscription-report', PrintSubscriptionPostMigrationReportAction()
        )
        workflow.get_path('test-all').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('test-all').insert_action(
            'print-report', PrintPostMigrationReportAction('test_all_report')
        )
        workflow.get_path('test-all').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_test_sites(workflow):
        workflow.add_entry_point('test-sites', EntryPointAction())
        workflow.get_path('test-sites').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('test-sites').insert_action(
            'test-sites', TestSitesAction()
        )
        workflow.get_path('test-sites').insert_action(
            'print-subscription-report', PrintSubscriptionPostMigrationReportAction()
        )
        workflow.get_path('test-sites').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('test-sites').insert_action(
            'print-report', PrintPostMigrationReportAction('test_sites_report')
        )
        workflow.get_path('test-sites').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_test_dns(workflow):
        workflow.add_entry_point('test-dns', EntryPointAction())
        workflow.get_path('test-dns').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('test-dns').insert_action(
            'test-dns', TestDnsAction()
        )
        workflow.get_path('test-dns').insert_action(
            'print-subscription-report', PrintSubscriptionPostMigrationReportAction()
        )
        workflow.get_path('test-dns').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('test-dns').insert_action(
            'print-report', PrintPostMigrationReportAction('test_dns_report')
        )
        workflow.get_path('test-dns').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_test_mail(workflow):
        workflow.add_entry_point('test-mail', EntryPointAction())
        workflow.get_path('test-mail').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('test-mail').insert_action(
            'test-mail', TestMailAction()
        )
        workflow.get_path('test-mail').insert_action(
            'print-subscription-report', PrintSubscriptionPostMigrationReportAction()
        )
        workflow.get_path('test-mail').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('test-mail').insert_action(
            'print-report', PrintPostMigrationReportAction('test_mail_report')
        )
        workflow.get_path('test-mail').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_test_users(workflow):
        workflow.add_entry_point('test-users', EntryPointAction())
        workflow.get_path('test-users').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('test-users').insert_action(
            'test-users', TestUsersAction()
        )
        workflow.get_path('test-users').insert_action(
            'print-subscription-report', PrintSubscriptionPostMigrationReportAction()
        )
        workflow.get_path('test-users').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('test-users').insert_action(
            'print-report', PrintPostMigrationReportAction('test_users_report')
        )
        workflow.get_path('test-users').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_test_databases(workflow):
        workflow.add_entry_point('test-databases', EntryPointAction())
        workflow.get_path('test-databases').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('test-databases').insert_action(
            'test-databases', TestDatabasesAction()
        )
        workflow.get_path('test-databases').insert_action(
            'print-subscription-report', PrintSubscriptionPostMigrationReportAction()
        )
        workflow.get_path('test-databases').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('test-databases').insert_action(
            'print-report', PrintPostMigrationReportAction('test_databases_report')
        )
        workflow.get_path('test-databases').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_test_services(workflow):
        workflow.add_entry_point('test-services', EntryPointAction())

        workflow.get_path('test-services').insert_action(
            'check-sources',
            workflow.get_shared_action('check-sources')
        )
        workflow.get_path('test-services').insert_action(
            'check-target',
            workflow.get_shared_action('check-target')
        )
        workflow.get_path('test-services').insert_action(
            'check-target-licenses',
            workflow.get_shared_action('check-target-licenses')
        )
        workflow.get_path('test-services').insert_action(
            'test-services', TestServicesAction()
        )
        workflow.get_path('test-services').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('test-services').insert_action(
            'print-report', PrintTestServicesReportAction()
        )
        workflow.get_path('test-services').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_list_ip_addresses(workflow):
        workflow.add_entry_point('list-ip-addresses', EntryPointAction())
        workflow.get_path('list-ip-addresses').insert_action(
            'list-ip-addresses', ListIPAddresses()
        )

    @staticmethod
    def _configure_entry_point_list_files_to_copy(workflow):
        workflow.add_entry_point('list-files-to-copy', EntryPointAction())
        workflow.get_path('list-files-to-copy').insert_actions(
            workflow.get_shared_action('prepare-assets').get_all_actions()
        )
        workflow.get_path('list-files-to-copy').insert_action(
            'list-files-to-copy', ListFiles()
        )
        workflow.get_path('list-files-to-copy').insert_action(
            'cleanup', workflow.get_shared_action('cleanup')
        )
        workflow.get_path('list-files-to-copy').register_overall_shutdown('cleanup')

    @staticmethod
    def _configure_entry_point_generate_html_log(workflow):
        workflow.add_entry_point('generate-html-log', EntryPointAction())
        workflow.get_path('generate-html-log').insert_action(
            'generate-html-log', GenerateHtmlLog()
        )

    @staticmethod
    def _configure_entry_point_remove_subscriptions_from_target(workflow):
        workflow.add_entry_point('remove-subscriptions-from-target', EntryPointAction())
        workflow.get_path('remove-subscriptions-from-target').insert_action(
            'check-migration-list', CheckMigrationListExistsForMigration()
        )
        workflow.get_path('remove-subscriptions-from-target').insert_action(
            'read-migration-list', workflow.get_shared_action('read-migration-list')
        )
        workflow.get_path('remove-subscriptions-from-target').insert_action(
            'remove-subscriptions-from-target', RemoveSubscriptionsFromTargetAction()
        )

    @staticmethod
    def _configure_entry_point_remove_subscriptions_from_queue(workflow):
        workflow.add_entry_point('remove-subscriptions-from-queue', EntryPointAction())
        workflow.get_path('remove-subscriptions-from-queue').insert_action(
            'check-migration-list', CheckMigrationListExistsForMigration()
        )
        workflow.get_path('remove-subscriptions-from-queue').insert_action(
            'read-migration-list', workflow.get_shared_action('read-migration-list')
        )
        workflow.get_path('remove-subscriptions-from-queue').insert_action(
            'remove-subscriptions-from-queue', RemoveQueueSubscriptionsAction()
        )
