# Copyright 1999-2012. Parallels IP Holdings GmbH. All Rights Reserved.
package PreMigrationChecks;

use strict;
use warnings;

use PreMigration;
use Dumper;
use ConfixxConfig;

sub checkOverall {
  my @supportedMaildrops = ("HOMEDIR/Maildir/", "HOMEDIR/Mailbox", "/var/spool/mail/USER");
  my $currentMaildrop = ConfixxConfig::getValue('maildrop');
  PreMigration::assert(
    !(grep { $_ eq  $currentMaildrop } @supportedMaildrops),
    'UNKNOWN_MAILDROP', {'maildrop' => $currentMaildrop}
  );
}

sub checkClient {
  my ($client, $clientInfo) = @_;

  my @sharedDomains = Dumper::getSharedDomains($client);
  PreMigration::assert(scalar(@sharedDomains) > 0, 'SHARED_DOMAINS', {'user' => $client, 'domains' => PreMigration::formatList(@sharedDomains)});

  my @atDomains = Dumper::getAtDomains($client);
  PreMigration::assert(scalar(@atDomains) > 0, 'AT_DOMAINS', {'user' => $client, 'atdomains' => PreMigration::formatList(@atDomains)});

  PreMigration::assert(Dumper::hasClientCustomErrorDocs($client), 'CLIENT_CUSTOM_ERROR_DOCS', {'user' => $client});

  my @emailsWithPercent = Dumper::getEmailsWithPercentSign($client);
  PreMigration::assert(scalar(@emailsWithPercent) > 0, 'EMAILS_WITH_PERCENT_SIGN', {'user'=> $client, 'emails' => PreMigration::formatList(@emailsWithPercent)});

  PreMigration::assert($clientInfo->{'shell'} == 0 && $clientInfo->{'scponly'} == 1, 'SCPONLY_NOT_SUPPORTED', {'user' => $client});
  PreMigration::assert($clientInfo->{'awstats'} == 1 && $clientInfo->{'statistik'} == 1, 'BOTH_AWSTATS_AND_WEBALIZER_ENABLED', {'user' => $client});

  PreMigration::assert(Dumper::hasClientCronJobs($client), 'CRON_JOBS', {'user' => $client});
  PreMigration::assert(Dumper::hasClientAddressbookEntries($client), 'ADDRESSBOOK', {'user' => $client});

  _checkPersonalInfo($client, $clientInfo);
}

sub checkReseller {
  my ($reseller, $resellerInfo) = @_;

  if (ConfixxConfig::getValue('dns_server')) {
    PreMigration::assert($resellerInfo->{'dnsspezial'} ne '', 'RESELLER_DNS_SPECIALS', {'user' => $reseller});
  }
  PreMigration::assert($resellerInfo->{'hilfecode'} ne '', 'RESELLER_HELP_MESSAGE', {'user' => $reseller});

  _checkPersonalInfo($reseller, $resellerInfo);
}

sub _checkPersonalInfo {
  my ($user, $info) = @_;

  PreMigration::assert(
    scalar($info->{'telefon'} && $info->{'telefon'} =~ /\//), 
    'PHONE_CONTAINS_SLASHES', {'user' => $user, 'phone' => $info->{'telefon'}},
    $info); 
  PreMigration::assert(
    scalar($info->{'fax'} && $info->{'fax'}  =~ /\//),
    'FAX_CONTAINS_SLASHES', {'user' => $user, 'fax' => $info->{'fax'}},
    $info); 

  _checkCustomPersonalInfo($user, $info);
    
  if ($info->{'land'} ne '') {
    my $countryCode = FormatConverter::convertCountry($info->{'land'});
    PreMigration::assert(!defined($countryCode), 'CAN_NOT_CONVERT_COUNTRY_CODE', {'user' => $user, 'country' => $info->{'land'}});
  }
    
  my $locale = FormatConverter::convertLocale($info->{'language'});

  # we don't know if it is really possible as there is no free-form language field
  PreMigration::assert(!defined($locale), 'CAN_NOT_CONVERT_LOCALE', {'user' => $user, 'language' => $info->{'language'}});
}

sub _checkCustomPersonalInfo {
  my ($user, $info) = @_;

  for (my $i = 1; $i <= 3; $i++) {
    if ($info->{"freifeldname$i"} ne '' || $info->{"freifeldname$i"} ne '') {
      PreMigration::message('PERSONAL_INFO_CUSTOM_FIELDS', {'user' => $user});
      return;
    }
  }
}

1;
