from parallels.core.content.mail.mailmigrate import CopyMailMailMigrator
from parallels.core.actions.utils.multithreading_properties import MultithreadingProperties
from parallels.core.migrator_config import MailContent
from parallels.plesk.source.custom import messages
import logging
from parallels.core.actions.base.subscription_action import SubscriptionAction
from parallels.core.content.mail.rsync import SourceMailDirectory, CopyMailRsync
from parallels.core.utils.paths.mail_paths import MailboxDirectory

logger = logging.getLogger(__name__)


class CopyMailMessages(SubscriptionAction):
    """Copy mail messages from custom panel's server - either with rsync (for Unix) or MailMigrator.exe (for Windows)"""

    def get_description(self):
        """Get short description of action as string

        :rtype: basestring
        """
        return messages.ACTION_COPY_MAIL_MESSAGES

    def get_failure_message(self, global_context, subscription):
        """Get message for situation when action failed

        :type global_context: parallels.common.global_context.GlobalMigrationContext
        :type subscription: parallels.common.migrated_subscription.MigratedSubscription
        :rtype: basestring
        """
        return messages.ACTION_COPY_MAIL_MESSAGES_FAILURE

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return False

    def get_multithreading_properties(self):
        """Get how multithreading should be applied for that action

        :rtype: parallels.core.actions.utils.multithreading_properties.MultithreadingProperties
        """
        return MultithreadingProperties(can_use_threads=True, use_threads_by_default=True)

    def filter_subscription(self, global_context, subscription):
        """Check if we should run this action on given subscription or not

        :type global_context: parallels.plesk.source.custom.global_context.CustomPanelGlobalMigrationContext
        :type subscription: parallels.plesk.source.custom.migrated_subscription.CustomPanelMigratedSubscription
        """
        # skip subscriptions w/o physical server
        if not subscription.get_source_info().is_server:
            return False

        if subscription.hosting_description is None or subscription.mail_source_server is None:
            return False

        if subscription.mail_source_server.mail_settings.mode == MailContent.NONE:
            return False

        return True

    def run(self, global_context, subscription):
        """Run action on given subscription

        :type global_context: parallels.plesk.source.custom.global_context.CustomPanelGlobalMigrationContext
        :type subscription: parallels.plesk.source.custom.migrated_subscription.CustomPanelMigratedSubscription
        """
        issues = []

        if subscription.get_source_info().is_windows:
            copy_mail = CopyMailMailMigrator()
        else:
            copy_mail = CopyMailRsync(CustomPanelSourceMailDirectory)
        copy_mail.copy_mail(global_context, subscription, issues)

        for issue in issues:
            global_context.safe.add_issue_subscription(subscription.name, issue)


class CustomPanelSourceMailDirectory(SourceMailDirectory):
    """Provide paths to directories with mail messages on source server according to the hosting description file"""

    def prepare(self):
        """
        :rtype: list[tuple(str, parallels.core.content.mail.rsync.MailHostingPath]]
        """
        directories = []

        for mailbox in self.subscription.hosting_description.iter_mailboxes():
            if mailbox.source_directory is None:
                continue
            if mailbox.domain_name.encode('idna') != self.domain.name.encode('idna'):
                continue

            directories.append((mailbox.source_directory, MailboxDirectory(mailbox.short_name)))

        return directories
