from contextlib import closing
import logging
import os

from parallels.core.dump.dump_archive_writer import DumpArchiveWriter
from parallels.core.utils.common import mkdir_p, open_no_inherit
from parallels.core.utils.pmm.agent import DumpAll
from parallels.plesk.source.helm3 import messages
from parallels.core import MigrationError
from parallels.plesk.source.helm3.helm3_agent import Helm3Agent


class BackupAgent(object):
    logger = logging.getLogger(__name__)

    def __init__(self, server):
        self.server = server 

    def make_backup(self, backup_filename, local_temp_directory, selection=DumpAll()):
        remote_log_filename = self.server.get_session_file_path('dump.log')
        local_log_filename = os.path.join(local_temp_directory, 'dump.log')
        remote_backup_xml_filename = self.server.get_session_file_path("backup.xml")
        local_backup_xml_filename = os.path.join(local_temp_directory, 'dump.xml')

        with self.server.runner() as runner:
            runner.remove_file(remote_log_filename)
            runner.remove_file(remote_backup_xml_filename)
            mkdir_p(local_temp_directory)

        try:
            helm_agent = Helm3Agent(self.server)
            args = ['--create-full-dump']

            filter_xml_contents = selection.get_filter_xml()
            if filter_xml_contents is not None:
                remote_filter_filename = self.server.get_session_file_path('filter.xml')
                with self.server.runner() as runner:
                    runner.upload_file_content(remote_filter_filename, filter_xml_contents)
                args.extend(["--filter", remote_filter_filename])

            try:
                helm_agent.run(args, redirect_output_file=remote_log_filename)
            finally:
                # Try to download log file anyway, even if there were issues
                with self.server.runner() as runner:
                    if runner.file_exists(remote_log_filename):
                        runner.get_file(remote_log_filename, local_log_filename)

            with self.server.runner() as runner:
                runner.get_file(remote_backup_xml_filename, local_backup_xml_filename)

            if os.path.exists(backup_filename):
                self.logger.debug(messages.LOG_REMOVE_EXISTING_BACKUP_FILE)
                os.remove(backup_filename)

            with \
                open_no_inherit(local_backup_xml_filename) as fp, \
                closing(DumpArchiveWriter(backup_filename)) as archive_writer:
                    archive_writer.add_dump(fp.read())
        except Exception as e:
            self.logger.debug(messages.LOG_EXCEPTION, exc_info=True)
            raise MigrationError(messages.CANNOT_GET_INFO_FROM_HELM % e)
