from parallels.common.hosting_check import \
	BaseHostingObject, BaseHostingCheckSource, ChildHostingObjectsList

class ServerBackupsCheckSource(BaseHostingCheckSource):
	"""Source for hosting checks - several Plesk backups
	
	Arguments:
	- backups - dictionary of Plesk backup objects, 
	value is from common.plesk_backup.plesk_backup_xml.load_backup
	key is a source server id
	- check_subscription_class - constructor of 
	HostingObjectSubscriptionBase subclass which provides concrete
	checks for a signle subscription
	"""
	def __init__(self, backups, check_subscription_class):
		self.backups = backups
		self.check_subscription_class = check_subscription_class

	def get_root_hosting_object(self):
		return HostingObjectServerList(self.backups, self.check_subscription_class)

class HostingObjectServerList(BaseHostingObject):
	"""Source for hosting checks - several servers represented by Plesk backups
	
	Arguments:
	- backups - dictionary of Plesk backup objects, 
	value is from common.plesk_backup.plesk_backup_xml.load_backup
	key is a source server id
	- check_subscription_class - constructor of 
	HostingObjectSubscriptionBase subclass which provides concrete
	checks for a signle subscription
	"""
	def __init__(self, backups, check_subscription_class):
		self.type = None
		self.name = None
		self.backups = backups
		self.check_subscription_class = check_subscription_class

	def get_child_hosting_objects(self):
		"""Child hosting objects: servers represented by Plesk backups"""
		result = ChildHostingObjectsList()
		for server_id, backup in self.backups.iteritems():
			result.child_hosting_objects.append(
				HostingObjectServer(
					server_id, backup, self.check_subscription_class
				)
			)
		return result

class HostingObjectServer(BaseHostingObject):
	"""Source for hosting checks - Plesk backup of a single server

	Arguments:
	- server_id - id of a source server (string)
	- backup - instance from common.plesk_backup.plesk_backup_xml.load_backup
	- check_subscription_class - constructor of 
	HostingObjectSubscriptionBase subclass which provides concrete
	checks for a signle subscription
	"""
	def __init__(self, server_id, backup, check_subscription_class):
		self.name = server_id
		self.type = 'Source server'
		self.backup = backup
		self.check_subscription_class = check_subscription_class

	def get_child_hosting_objects(self):
		"""Child hosting objects: resellers, admin's clients and subscriptions"""
		result = ChildHostingObjectsList()
		for reseller in self.backup.iter_resellers():
			result.child_hosting_objects.append(
				HostingObjectReseller(
					reseller, self.backup, self.check_subscription_class
				)
			)
		for client in self.backup.iter_clients():
			result.child_hosting_objects.append(
				HostingObjectClient(
					client, self.backup, self.check_subscription_class
				)
			)
		for subscription in self.backup.iter_admin_subscriptions():
			result.child_hosting_objects.append(
				self.check_subscription_class(self.backup, subscription)
			)
		return result

class HostingObjectReseller(BaseHostingObject):
	"""Source for hosting checks - reseller from Plesk backup
	
	- reseller - Reseller object from common.plesk_backup_data_model
	- backup - instance from common.plesk_backup.plesk_backup_xml.load_backup
	- check_subscription_class - constructor of 
	HostingObjectSubscriptionBase subclass which provides concrete
	checks for a signle subscription
	"""
	def __init__(self, reseller, backup, check_subscription_class):
		self.name = reseller.login
		self.type = 'Reseller'
		self.reseller = reseller
		self.backup = backup
		self.check_subscription_class = check_subscription_class

	def get_child_hosting_objects(self):
		"""Child hosting objects: reseller's clients and subscriptions"""
		result = ChildHostingObjectsList()
		for client in self.reseller.clients:
			result.child_hosting_objects.append(
				HostingObjectClient(
					client, self.backup, self.check_subscription_class
				)
			)
		for subscription in self.reseller.subscriptions:
			result.child_hosting_objects.append(
				self.check_subscription_class(self.backup, subscription)
			)
		return result

class HostingObjectClient(BaseHostingObject):
	"""Source for hosting checks - client from Plesk backup

	- client - Client object from common.plesk_backup_data_model
	- backup - instance from common.plesk_backup.plesk_backup_xml.load_backup
	- check_subscription_class - constructor of 
		HostingObjectSubscriptionBase subclass which provides concrete
		checks for a signle subscription
	"""
	def __init__(self, client, backup, check_subscription_class):
		self.name = client.login
		self.type = 'Client'
		self.client = client
		self.backup = backup
		self.check_subscription_class = check_subscription_class

	def get_child_hosting_objects(self):
		"""Child hosting objects: client's subscriptions"""
		result = ChildHostingObjectsList()
		for subscription in self.client.subscriptions:
			result.child_hosting_objects.append(
				self.check_subscription_class(self.backup, subscription)
			)
		return result

class HostingObjectSubscriptionBase(BaseHostingObject):
	"""Source for hosting checks - subscription from Plesk backup
	
	Override for various services (web, mail, databases, DNS, etc)
	
	- subscription - Subscription object from common.plesk_backup_data_model
	- backup - instance from common.plesk_backup.plesk_backup_xml.load_backup
	"""
	def __init__(self, backup, subscription):
		self.name = subscription.name
		self.type = 'Subscription'
		self.backup = backup
		self.subscription = subscription
