import logging

from parallels.common.actions.base.subscription_action import SubscriptionAction
from parallels.common import checking
from parallels.common.actions.utils.multithreading_properties import MultithreadingProperties
from parallels.common.utils import plesk_api_utils
from parallels.common.import_api.import_api import UserAlreadyExistsException, LoginIsInvalidException
from parallels.common.target_panels import TargetPanels

logger = logging.getLogger(__name__)


class RestoreSysuserLogins(SubscriptionAction):
	"""PPA does create webspace with system user login like sub83_54. In source
	backup, corresponding system user's login is quite different, like
	john_doe. This procedure tries to update PPA's system user login to what is
	in source backup, and either succeeds => then login is changed in PPA or
	fails because the desired login is already used elsewhere => then it will
	use auto-generated login. It will be updated in converted backup by
	ChangeSysuserLogin action.
	"""
	def get_description(self):
		return "Restore logins of system users"

	def get_failure_message(self, global_context, subscription):
		"""
		:type global_context: parallels.common.global_context.GlobalMigrationContext
		:type subscription: parallels.common.migrated_subscription.MigratedSubscription
		"""
		return "Failed to restore login of system user for subscription '%s'" % (subscription.name)

	def filter_subscription(self, global_context, subscription):
		"""
		:type global_context: parallels.common.global_context.GlobalMigrationContext
		:type subscription: parallels.common.migrated_subscription.MigratedSubscription
		"""
		return True

	def get_multithreading_properties(self):
		return MultithreadingProperties(can_use_threads=True)

	def run(self, global_context, subscription):
		"""
		:type global_context: parallels.common.global_context.GlobalMigrationContext
		:type subscription: parallels.common.migrated_subscription.MigratedSubscription
		"""
		with global_context.safe.try_subscription(subscription.name, u"Failed to restore subscription system user's login."):
			sysuser_name = subscription.raw_backup.phosting_sysuser_name
			if sysuser_name is None:
				logger.debug("There is no system user login in backup, skip changing")
				return

			if global_context.target_panel == TargetPanels.PPA:
				sysuser_name = sysuser_name.replace('.', '_')

			ppa_sysuser_name = plesk_api_utils.get_subscription_sysuser_name(
				subscription.panel_target_server.plesk_api(), subscription.name
			)

			if ppa_sysuser_name is None:
				# subscription has no system user on target
				# now we simply ignore that case, but probably it would be
				# better to handle such situation better to avoid issues
				# on the next steps of migration
				logger.debug("There is no system user login on target system, skip changing")
				return

			if ppa_sysuser_name != sysuser_name:
				try:
					global_context.import_api.change_sysuser_login(
						subscription.name, sysuser_name
					)
				except UserAlreadyExistsException:
					subscription.add_report_issue(
						global_context.pre_check_report,
						checking.Problem(
							'unable_to_restore_original_sysuser_name_already_exists', checking.Problem.WARNING, 
							u"Unable to restore the original username for the system user of the subscription %s: a system user named %s is already present in PPA. For this reason, the subscription owner will not be able to log in to FTP under the original username." % (subscription.name, sysuser_name)
						),
						u"Notify the subscription owner of the new FTP username: %s. The subscription owner can change this username using the PPA Hosting Panel." % ppa_sysuser_name
					)
				except LoginIsInvalidException:
					subscription.add_report_issue(
						global_context.pre_check_report,
						checking.Problem(
							'unable_to_restore_original_sysuser_name_does_not_conform_to_ppa_rules', checking.Problem.WARNING, 
							u"Unable to restore the original username ('%s') for the system user of the subscription %s: it does not conform to the PPA rules. For this reason, the subscription owner will not be able to log in to FTP under the original username." % (sysuser_name, subscription.name)
						),
						u"Notify the subscription owner of the new FTP username: %s. The subscription owner can change this username using the PPA Hosting Panel." % ppa_sysuser_name
					)
