import errno
import ntpath
import os
import shutil
from contextlib import contextmanager

from parallels.core import local_command_exec, messages
from parallels.core.runners.exceptions.mssql import MSSQLException
from parallels.core.runners.windows.base import WindowsRunner
from parallels.core.utils.common import cached, mkdir_p, default, open_no_inherit
from parallels.core.utils.common.ip import resolve, resolve_all
from parallels.core.utils.common.logging import create_safe_logger
from parallels.core.utils.file_utils import iter_full_file_paths
from parallels.core.utils.steps_profiler import get_default_steps_profiler

logger = create_safe_logger(__name__)
profiler = get_default_steps_profiler()


class LocalWindowsRunner(WindowsRunner):
    """Execute commands on local Windows server"""

    def __init__(self, disable_profiling=False):
        super(LocalWindowsRunner, self).__init__(host_description='the local server', hostname=None)
        self._disable_profiling = disable_profiling

    @staticmethod
    @cached
    def get_instance(disable_profiling=False):
        return LocalWindowsRunner(disable_profiling)

    def _run_command_and_decode_output(self, cmd_str, output_codepage=None, error_policy='strict', env=None):
        codepage = output_codepage if output_codepage else self.codepage
        return local_command_exec(cmd_str, None, codepage, error_policy, env)

    def _sh_unchecked_no_logging(
        self, cmd_str, args=None, stdin_content=None, output_codepage=None,
        error_policy='strict', env=None, log_output=True, working_dir=None, redirect_output_file=None
    ):
        if redirect_output_file is not None:
            raise NotImplementedError()

        codepage = output_codepage if output_codepage else self.codepage
        command_str = self._format_sh_command(cmd_str, args)
        return local_command_exec(command_str, stdin_content, codepage, error_policy, env, working_dir)

    def upload_file(self, local_filename, remote_filename):
        shutil.copy2(local_filename, remote_filename)

    def upload_file_content(self, filename, content):
        with open_no_inherit(filename, "wb") as fp:
            fp.write(content)

    def get_file(self, remote_filename, local_filename):
        if local_filename != remote_filename:
            logger.debug(messages.DEBUG_COPY_LOCAL_FILE, remote_filename, local_filename)
            shutil.copy2(remote_filename, local_filename)

    def get_file_contents(self, remote_filename):
        with open_no_inherit(remote_filename, 'rb') as fp:
            return fp.read().decode('utf-8')

    def get_file_size(self, remote_filename):
        return os.path.getsize(remote_filename)

    def move(self, src_path, dst_path):
        shutil.move(src_path, dst_path)

    def get_files_list(self, path):
        return os.listdir(path)

    def iter_files_list_nested(self, path):
        """Iterate over all (including files in nested directories) file names inside a path"""
        return iter_full_file_paths(path)

    def mkdir(self, dirname):
        mkdir_p(dirname)

    def remove_directory(self, directory, is_remove_root=True):
        for root, dirs, files in os.walk(directory):
            for name in files:
                self.remove_file(ntpath.join(directory, name))
            for name in dirs:
                self.remove_directory(ntpath.join(directory, name))
            if is_remove_root:
                os.rmdir(root)

    def remove_file(self, filename):
        try:
            os.remove(filename)
        except OSError as e:
            # if file does not exist - just skip
            if e.errno != errno.ENOENT:
                raise

    def check(self, server_description):
        # No need to check local execution, it should never fail
        pass

    def file_exists(self, filename):
        return os.path.exists(filename)

    def mssql_execute(self, connection_settings, query, args=None):
        """Execute MSSQL query, do not return any results

        :type connection_settings: parallels.core.runners.entities.MSSQLConnectionSettings
        :type query: str | unicode
        :type args: dict | None
        :raises parallels.core.runners.exceptions.mssql.MSSQLException:
        :rtype: None
        """
        import pymssql
        self._log_mssql_query(connection_settings, query, args)
        with self._wrap_pymssql_error(connection_settings, query, args):
            host, port = self._split_mssql_host_port(connection_settings.host)
            with pymssql.connect(
                host, connection_settings.user, connection_settings.password,
                default(connection_settings.database, ''), port=port
            ) as conn:
                # Disable autocommit, as some commands like backup do not work when it is enabled
                conn.autocommit(True)
                try:
                    with conn.cursor() as cursor:
                        cursor.execute(query, args)
                finally:
                    conn.autocommit(False)

    def mssql_query(self, connection_settings, query, args=None):
        """Execute MSSQL query, return results as a list of dictionaries

        :type connection_settings: parallels.core.runners.entities.MSSQLConnectionSettings
        :type query: str | unicode
        :type args: dict | None
        :raises parallels.core.runners.exceptions.mssql.MSSQLException:
        :rtype: list[dict]
        """
        import pymssql
        self._log_mssql_query(connection_settings, query, args)
        with self._wrap_pymssql_error(connection_settings, query, args):
            host, port = self._split_mssql_host_port(connection_settings.host)
            with pymssql.connect(
                host, connection_settings.user, connection_settings.password,
                default(connection_settings.database, ''), port=port
            ) as conn:
                with conn.cursor(as_dict=True) as cursor:
                    cursor.execute(query, args)
                    result = cursor.fetchall()
                    self._log_mssql_query_result(result)
                    return result

    def resolve(self, address):
        """Resolve hostname into IP address.

        Return the first IP address, or None if we were not able to resolve.
        If address is already a valid IP address - return it as is.

        :type address: str | unicode
        :rtype: None | str | unicode
        """
        return resolve(address)

    def resolve_all(self, address):
        """Resolve hostname into list of IP addresses.

        If address is already a valid IP address - return
        list that contains only that address.
        Otherwise resolve and return list of IP addresses
        (IPv6 and IPv4 all in one list).
        If we were not able to resolve, empty list is returned.

        :type address: str | unicode
        :rtype: list[str | unicode]
        """
        return resolve_all(address)

    @contextmanager
    def _wrap_pymssql_error(self, connection_settings, query, args):
        """Context manager, within its scope all pymssql errors will be wrapped with MSSQLException

        :type connection_settings: parallels.core.runners.entities.MSSQLConnectionSettings
        :type query: str | unicode
        :type args: dict | None
        """
        import pymssql
        try:
            yield
        except pymssql.Error as e:
            logger.debug(messages.LOG_EXCEPTION, exc_info=True)
            if len(e.args) > 0 and isinstance(e.args[0], basestring):
                raise MSSQLException(
                    code=None, message=e.args[0],
                    connection_settings=connection_settings, query=query, args=args,
                    connected_from=self.host_description
                )
            elif len(e.args) > 0 and len(e.args[0]) > 1:
                raise MSSQLException(
                    code=e.args[0][0], message=e.args[0][1],
                    connection_settings=connection_settings, query=query, args=args,
                    connected_from=self.host_description
                )
            else:
                raise MSSQLException(
                    code=None, message=unicode(e),
                    connection_settings=connection_settings, query=query, args=args,
                    connected_from=self.host_description
                )

    @staticmethod
    def _split_mssql_host_port(hostname):
        """Split MSSQL hostname into host and port parts

        For 'localhost,1435' returns ('localhost', '1435').
        For 'localhost' returns ('localhost', None).

        :type hostname: str | unicode
        :rtype: tuple
        """
        parts = hostname.split(',', 1)
        if len(parts) == 2:
            return parts[0], parts[1]
        else:
            return hostname, '1433'
