from parallels.plesk.source.plesk import messages
import logging
import posixpath

from parallels.core.actions.base.subscription_action import SubscriptionAction
from parallels.core.utils import subscription_filter

logger = logging.getLogger(__name__)


class FixUnixPermissions(SubscriptionAction):
    """Fix file ownership for files in domain directory after copying files with original permissions with rsync"""
    def get_description(self):
        """Get short description of action as string

        :rtype: str
        """
        return messages.ACTION_FIX_UNIX_PERMISSIONS_DESCRIPTION

    def get_failure_message(self, global_context, subscription):
        """Get message for situation when action failed

        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        :rtype: basestring
        """
        return messages.ACTION_FIX_UNIX_PERMISSIONS_FAILED % subscription.name

    def is_critical(self):
        """If action is critical or not

        If action is critical and it failed for a subscription, migration tool
        won't run the next operations for the subscription.

        :rtype: bool
        """
        return False

    def filter_subscription(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        return subscription_filter.unix_with_virtual_hosting(
            subscription
        )

    def run(self, global_context, subscription):
        """
        :type global_context: parallels.core.global_context.GlobalMigrationContext
        :type subscription: parallels.core.migrated_subscription.MigratedSubscription
        """
        sysuser_name = subscription.converted_dump.get_phosting_sysuser_name()
        if sysuser_name is None:
            return

        logger.debug(messages.LOG_FIX_FILE_OWNERSHIP_AFTER_MOVING)

        target_vhost_dir = posixpath.join(
            subscription.web_target_server.vhosts_dir, 
            subscription.name_idn
        )

        with subscription.web_target_server.runner() as runner:
            # Set ownership for all files inside virtual host directory
            runner.sh('/bin/chown -R {username}:psacln {vhost_dir}', dict(
                username=sysuser_name, vhost_dir=target_vhost_dir
            ))
            # Set ownership for virtual host directory itself - it should have 'psaserv' as group owner
            runner.sh('/bin/chgrp psaserv {vhost_dir}', dict(
                vhost_dir=target_vhost_dir
            ))
            # Each document root should have 'psaserv' as group owner
            for domain in subscription.converted_dump.iter_domains():
                www_root_path = posixpath.join(target_vhost_dir, domain.www_root)
                cgi_bin_path = posixpath.join(target_vhost_dir, domain.www_root, 'cgi-bin')
                for filename in (www_root_path, cgi_bin_path):
                    runner.sh(
                        "test ! -e {filename} || test -L {filename}"
                        "|| /bin/chgrp psaserv {filename}", dict(filename=filename)
                    )

            # Fix permissions for 'logs' directory, otherwise statistics calculation may get broken
            runner.sh(
                "test ! -e {filename} || test -L {filename}"
                "|| /bin/chown {username}:root {filename}", dict(
                    username=sysuser_name,
                    filename=posixpath.join(target_vhost_dir, 'logs')
                )
            )
            runner.sh(
                "test ! -e {filename} || test -L {filename}"
                "|| /bin/chmod 700 {filename}", dict(
                    filename=posixpath.join(target_vhost_dir, 'logs')
                )
            )