from collections import defaultdict
from contextlib import closing
from StringIO import StringIO
from xml.etree import ElementTree
import logging
import os
import re

from parallels.core.dump.dump_archive_writer import DumpArchiveWriter
from parallels.core.dump.file_adapters import TarFileAdapter
from parallels.core.dump.file_adapters import ZipFileAdapter
from parallels.core.registry import Registry
from parallels.core.reports.model.issue import Issue
from parallels.core.utils.common import safe_format
from parallels.core.utils.common.xml import xml_to_string_pretty
from parallels.plesk.source.plesk import messages

logger = logging.getLogger(__name__)


class PleskHostingDescriptionConverter(object):
    """
    :type _result_dump_file: str | unicode
    :type _result_dump_writer: parallels.core.dump.dump_archive_writer.DumpArchiveWriter
    :type _dump_reader: parallels.core.dump.file_adapters.ArchiveAdapter
    :type _extra_dump_dir: str | unicode
    :type _extra_dump_prefix: str | unicode
    :type _extra_dump_timestamp: str | unicode
    :type _extra_dump_entities: dict[str|unicode, dict[str|unicode, str|unicode]]
    :type _xs_types: dict[str|unicode, str|unicode]
    :type _schema_namespaces: dict[str|unicode, str|unicode]
    :type _schema: xml.etree.ElementTree.ElementTree
    :type _identities: dict
    """
    def __init__(self, result_dump_file, dump_file, extra_dump_dir, extra_dump_prefix, extra_dump_timestamp):
        """
        :type result_dump_file: str | unicode
        :type dump_file: str | unicode
        :type extra_dump_dir: str | unicode
        :type extra_dump_prefix: str | unicode
        :type extra_dump_timestamp: str | unicode
        """
        self._result_dump_file = result_dump_file
        self._result_dump_writer = None
        self._dump_reader = TarFileAdapter(dump_file) if dump_file.endswith('.tar') else ZipFileAdapter(dump_file)
        self._extra_dump_dir = extra_dump_dir
        self._extra_dump_prefix = extra_dump_prefix
        self._extra_dump_timestamp = extra_dump_timestamp
        self._extra_dump_entities = defaultdict(dict)
        xs_namespace = 'http://www.w3.org/2001/XMLSchema'
        self._xs_types = {
            name: '{%s}%s' % (xs_namespace, name)
            for name in ['element', 'sequence', 'choice', 'all', 'complexContent']
        }
        self._schema_namespaces = {'xs': xs_namespace}
        self._schema = ElementTree.ElementTree(ElementTree.Element(None))
        self._identities = {
            'domain-alias': lambda e: e.get('name'),
            'dns-zone-param': lambda e: e.get('name'),
            'dnsrec': lambda e: (e.get('type'), e.get('src'), e.get('dst')),
            'custom-button': lambda e: (e.get('interface-place'), e.get('text')),
            'plan': lambda e: e.get('plan-guid'),
            'db-server': lambda e: (e.get('type'), e.findtext('host'), e.findtext('port')),
            'database': lambda e: (e.get('name'), e.get('type'), e.findtext('db-server/host'), e.findtext('db-server/port')),
            'dbuser': lambda e: (e.get('name'), e.findtext('db-server/host'), e.findtext('db-server/port')),
            'limit': lambda e: e.get('name'),
            'permission': lambda e: e.get('name'),
            'description': lambda e: (e.get('object-name'), e.get('object-type')),
            'ip': lambda e: e.findtext('ip-address'),
            'param': lambda e: e.findtext('name'),
            'parameter': lambda e: e.findtext('name'),
            'setting': lambda e: e.findtext('name'),
            'property': lambda e: e.findtext('name'),
            'mailuser': lambda e: e.get('name'),
            'maillist': lambda e: e.get('name').lower() if e.get('name') else e.get('name'),
            'certificate': lambda e: e.get('name'),
            'wordpress-instance': lambda e: e.get('path'),
            'composer-instance': lambda e: e.get('path'),
            'pdir': lambda e: e.get('name'),
            'pduser': lambda e: e.get('name'),
            'mime-type': lambda e: e.get('ext'),
            'webuser': lambda e: e.get('name'),
            'ftpuser': lambda e: e.get('name'),
            'subdomain': lambda e: e.get('name'),
            'site': lambda e: e.get('name'),
            'sapp-installed': lambda e: e.findtext('sapp-installdir/sapp-prefix'),
            'sapp-entry-point': lambda e: e.findtext('label'),
            'forwarding': lambda e: e.text,
            'attach': lambda e: e.get('file'),
            'autoresponder-limit': lambda e: e.get('name'),
            'anonftp-limit': lambda e: e.get('name'),
            'anonftp-permission': lambda e: e.get('name'),
            'extension': lambda e: e.text,
            'friend': lambda e: e.text,
            'owner': lambda e: e.text.lower() if e.text else e.text,
            'recipient': lambda e: e.text.lower() if e.text else e.text,
            'blacklist-member': lambda e: e.text,
            'whitelist-member': lambda e: e.text,
            'trusted-locale': lambda e: e.text,
            'trusted-language': lambda e: e.text,
            'trusted-network': lambda e: e.text,
            'related-site': lambda e: e.get('name'),
        }

    def load_schema(self, schema_path):
        """
        :type schema_path: str | unicode
        """
        self._schema = self._load_schema(schema_path)

    def merge(self):
        self._index_extra_dump_entities()
        self._merge_entities()

    def merge_dumps(self, dump, extra_dump):
        """
        :type dump: xml.etree.ElementTree.ElementTree
        :type extra_dump: xml.etree.ElementTree.ElementTree
        :rtype: xml.etree.ElementTree.ElementTree
        """
        merged_dump = ElementTree.ElementTree(ElementTree.Element(dump.getroot().tag))
        schema_root_node = self._schema.find("xs:element[@name='%s']" % dump.getroot().tag, self._schema_namespaces)
        self._merge_xml_elements(
            merged_dump.getroot(),
            dump.getroot(),
            extra_dump.getroot(),
            self._get_xml_element_schema(schema_root_node)
        )
        return merged_dump

    def _merge_entities(self):
        self._result_dump_writer = DumpArchiveWriter(self._result_dump_file)
        with closing(self._result_dump_writer):
            reseller_name, client_name, domain_name = None, None, None
            for dump_entry in self._dump_reader.get_names():
                if not self._is_xml_file(dump_entry):
                    continue
                dump_xml = self._dump_reader.extract(dump_entry).read()
                dump = ElementTree.parse(StringIO(dump_xml))
                guid = None
                reseller_node, client_node, domain_node = dump.find('reseller'), None, None
                if reseller_node is not None:
                    reseller_name, client_name, domain_name = reseller_node.get('name'), None, None
                    guid = reseller_node.get('guid')
                else:
                    if re.search('^resellers/', dump_entry) is None:
                        reseller_name = None
                    client_node = dump.find('client')
                if client_node is not None:
                    client_name, domain_name = client_node.get('name'), None
                    guid = client_node.get('guid')
                else:
                    if re.search('^(resellers/[^/]+/)?clients/', dump_entry) is None:
                        client_name = None
                    domain_node = dump.find('domain')
                if domain_node is not None:
                    domain_name = domain_node.get('name')
                    guid = domain_node.get('guid')
                    merged_dump_xml = self._merge_domain(domain_name, dump)
                    if merged_dump_xml:
                        dump_xml = merged_dump_xml
                else:
                    domain_name = None
                self._result_dump_writer.add_dump(
                    dump_xml, reseller_name=reseller_name, client_name=client_name, domain_name=domain_name, guid=guid
                )

    def _merge_domain(self, domain_name, dump):
        """
        :type domain_name: str | unicode
        :type dump: xml.etree.ElementTree.ElementTree
        :rtype: str | unicode | None
        """
        extra_dump_file = self._get_extra_dump_file('domain', domain_name)
        if extra_dump_file is None:
            Registry.get_instance().get_context().pre_check_report.add_issue(
                'hosting_description_merge_domain_not_found',
                Issue.SEVERITY_WARNING,
                safe_format(messages.DOMAIN_CONFIGURATION_BACKUP_IS_NOT_FOUND, name=domain_name)
            )
            return None
        try:
            extra_dump = ElementTree.parse(extra_dump_file)
            merged_dump = self.merge_dumps(dump, extra_dump)
            return xml_to_string_pretty(merged_dump)
        except Exception as e:
            logger.debug(messages.LOG_EXCEPTION, exc_info=True)
            Registry.get_instance().get_context().pre_check_report.add_issue(
                'hosting_description_merge_domain_failed',
                Issue.SEVERITY_WARNING,
                safe_format(messages.FAILED_TO_MERGE_DOMAIN_CONFIGURATION_BACKUP, name=domain_name, error=unicode(e))
            )

    def _merge_xml_elements(self, merged_dump_node, dump_node, extra_dump_node, dump_node_schema):
        """
        :type merged_dump_node: xml.etree.ElementTree.Element
        :type dump_node: xml.etree.ElementTree.Element
        :type extra_dump_node: xml.etree.ElementTree.Element
        :type dump_node_schema: xml.etree.ElementTree.Element | None
        """
        if extra_dump_node.tag in ['content', 'extensions']:
            # skip the 'content' elements of the extra dump because currently we
            # merge XML files only and there is no necessity to merge archives;
            # skip the 'extensions' elements of the extra dump because extension data
            # will be deployed by separate actions
            extra_dump_node = ElementTree.Element(None)

        self._merge_xml_attributes(merged_dump_node, dump_node, extra_dump_node)

        # Our xml_to_string_pretty function do not process XML formatting symbols ('\n', '\t') correctly.
        # So we skip such symbols here.
        if dump_node.text is not None and (dump_node.text == '' or dump_node.text.strip()):
            merged_dump_node.text = dump_node.text
        elif extra_dump_node.text is not None and (extra_dump_node.text == '' or extra_dump_node.text.strip()):
            merged_dump_node.text = extra_dump_node.text

        if dump_node_schema is not None:
            self._merge_xml_elements_with_schema(merged_dump_node, dump_node, extra_dump_node, dump_node_schema)
        else:
            self._merge_xml_elements_without_schema(merged_dump_node, dump_node, extra_dump_node)

    def _merge_xml_attributes(self, merged_dump_node, dump_node, extra_dump_node):
        """
        :type merged_dump_node: xml.etree.ElementTree.Element
        :type dump_node: xml.etree.ElementTree.Element
        :type extra_dump_node: xml.etree.ElementTree.Element
        """
        for name, value in dump_node.items():
            merged_dump_node.set(name, value)
        for name, value in extra_dump_node.items():
            if merged_dump_node.get(name) is None:
                merged_dump_node.set(name, value)

    def _merge_xml_elements_with_schema(self, merged_dump_node, dump_node, extra_dump_node, dump_node_schema):
        """Merge using XSD-schema. Fix cases when sequence of elements is wrong or there are unwanted elements.
        :type merged_dump_node: xml.etree.ElementTree.Element
        :type dump_node: xml.etree.ElementTree.Element
        :type extra_dump_node: xml.etree.ElementTree.Element
        :type dump_node_schema: xml.etree.ElementTree.Element
        """
        # 1. Get next child element from schema.
        for node_name, schema_node in self._iter_xml_element_schema_sequence(dump_node_schema):
            # 2. Look for and merge the same elements from extra dump.
            for extra_dump_child_node in extra_dump_node.iterfind(node_name):
                dump_child_node_index = self._find_child_xml_element(dump_node, extra_dump_child_node)
                if dump_child_node_index != -1:
                    dump_child_node = dump_node[dump_child_node_index]
                else:
                    dump_child_node = ElementTree.Element(None)
                merged_dump_child_node = ElementTree.Element(extra_dump_child_node.tag)
                merged_dump_node.append(merged_dump_child_node)
                self._merge_xml_elements(
                    merged_dump_child_node,
                    dump_child_node,
                    extra_dump_child_node,
                    schema_node
                )
            # 3. Look for and copy the same rest elements from dump.
            for dump_child_node in dump_node.iterfind(node_name):
                extra_dump_child_node_index = self._find_child_xml_element(extra_dump_node, dump_child_node)
                if extra_dump_child_node_index == -1:
                    merged_dump_child_node = ElementTree.Element(dump_child_node.tag)
                    merged_dump_node.append(merged_dump_child_node)
                    self._merge_xml_elements(
                        merged_dump_child_node,
                        dump_child_node,
                        ElementTree.Element(None),
                        schema_node
                    )

    def _merge_xml_elements_without_schema(self, merged_dump_node, dump_node, extra_dump_node):
        """Merge without using XSD-schema. It is required for elements with 'any' type or when schema was not found.
        :type merged_dump_node: xml.etree.ElementTree.Element
        :type dump_node: xml.etree.ElementTree.Element
        :type extra_dump_node: xml.etree.ElementTree.Element
        """
        dump_index, extra_dump_index = 0, 0
        while dump_index < len(dump_node) or extra_dump_index < len(extra_dump_node):
            if extra_dump_index < len(extra_dump_node):
                # 1. Get next child element from extra dump.
                extra_dump_child_node = extra_dump_node[extra_dump_index]
                # 2. Look for the same element in dump.
                dump_child_node_index = self._find_child_xml_element(dump_node, extra_dump_child_node)
                if dump_child_node_index != -1:
                    # 3. Merge founded element and all previous elements from dump and then go to step 1.
                    while dump_index <= dump_child_node_index:
                        dump_child_node = dump_node[dump_index]
                        extra_dump_child_node_index = self._find_child_xml_element(extra_dump_node, dump_child_node)
                        if extra_dump_child_node_index == -1 or extra_dump_child_node_index >= extra_dump_index:
                            if extra_dump_child_node_index != -1:
                                extra_dump_child_node = extra_dump_node[extra_dump_child_node_index]
                            else:
                                extra_dump_child_node = ElementTree.Element(None)
                            merged_dump_child_node = ElementTree.Element(dump_child_node.tag)
                            merged_dump_node.append(merged_dump_child_node)
                            self._merge_xml_elements(
                                merged_dump_child_node,
                                dump_child_node,
                                extra_dump_child_node,
                                None
                            )
                        dump_index += 1
                    extra_dump_index += 1
                    continue
                if dump_index < len(dump_node):
                    # 4. Get next child element from dump.
                    dump_child_node = dump_node[dump_index]
                    # 5. Look for the same element in extra dump.
                    extra_dump_child_node_index = self._find_child_xml_element(extra_dump_node, dump_child_node)
                    if extra_dump_child_node_index != -1:
                        # 6. Merge founded element and all previous elements from extra dump and then go to step 1.
                        while extra_dump_index <= extra_dump_child_node_index:
                            extra_dump_child_node = extra_dump_node[extra_dump_index]
                            dump_child_node_index = self._find_child_xml_element(dump_node, extra_dump_child_node)
                            if dump_child_node_index == -1 or dump_child_node_index >= dump_index:
                                if dump_child_node_index != -1:
                                    dump_child_node = dump_node[dump_child_node_index]
                                else:
                                    dump_child_node = ElementTree.Element(None)
                                merged_dump_child_node = ElementTree.Element(extra_dump_child_node.tag)
                                merged_dump_node.append(merged_dump_child_node)
                                self._merge_xml_elements(
                                    merged_dump_child_node,
                                    dump_child_node,
                                    extra_dump_child_node,
                                    None
                                )
                            extra_dump_index += 1
                        dump_index += 1
                        continue
                # 7. Copy child element from extra dump and then go to step 1.
                merged_dump_child_node = ElementTree.Element(extra_dump_child_node.tag)
                merged_dump_node.append(merged_dump_child_node)
                self._merge_xml_elements(
                    merged_dump_child_node,
                    ElementTree.Element(None),
                    extra_dump_child_node,
                    None
                )
                extra_dump_index += 1
                continue

            # 8. Copy rest elements from dump.
            dump_child_node = dump_node[dump_index]
            extra_dump_child_node_index = self._find_child_xml_element(extra_dump_node, dump_child_node)
            if extra_dump_child_node_index == -1 or extra_dump_child_node_index >= extra_dump_index:
                merged_dump_child_node = ElementTree.Element(dump_child_node.tag)
                merged_dump_node.append(merged_dump_child_node)
                self._merge_xml_elements(
                    merged_dump_child_node,
                    dump_child_node,
                    ElementTree.Element(None),
                    None
                )
            dump_index += 1

    def _index_extra_dump_entities(self):
        resellers_dump_dir = os.path.join(self._extra_dump_dir, 'resellers')
        if os.path.exists(resellers_dump_dir):
            self._index_resellers(resellers_dump_dir)
        clients_dump_dir = os.path.join(self._extra_dump_dir, 'clients')
        if os.path.exists(clients_dump_dir):
            self._index_clients(clients_dump_dir)
        domains_dump_dir = os.path.join(self._extra_dump_dir, 'domains')
        if os.path.exists(domains_dump_dir):
            self._index_domains(domains_dump_dir)

    def _index_resellers(self, dump_dir):
        """
        :type dump_dir: str|unicode
        """
        for dir_entry in os.listdir(dump_dir):
            dir_entry_path = os.path.join(dump_dir, dir_entry)
            if os.path.isdir(dir_entry_path):
                self._index_reseller(dir_entry_path)

    def _index_reseller(self, dump_dir):
        """
        :type dump_dir: str|unicode
        """
        for dir_entry in os.listdir(dump_dir):
            dir_entry_path = os.path.join(dump_dir, dir_entry)
            if os.path.isdir(dir_entry_path):
                if dir_entry == 'clients':
                    self._index_clients(dir_entry_path)
                elif dir_entry == 'domains':
                    self._index_domains(dir_entry_path)
                continue
            if self._is_extra_dump_file(dir_entry):
                self._index_entity('reseller', dir_entry_path)

    def _index_clients(self, dump_dir):
        """
        :type dump_dir: str|unicode
        """
        for dir_entry in os.listdir(dump_dir):
            dir_entry_path = os.path.join(dump_dir, dir_entry)
            if os.path.isdir(dir_entry_path):
                self._index_client(dir_entry_path)

    def _index_client(self, dump_dir):
        """
        :type dump_dir: str|unicode
        """
        for dir_entry in os.listdir(dump_dir):
            dir_entry_path = os.path.join(dump_dir, dir_entry)
            if os.path.isdir(dir_entry_path):
                if dir_entry == 'domains':
                    self._index_domains(dir_entry_path)
                continue
            if self._is_extra_dump_file(dir_entry):
                self._index_entity('client', dir_entry_path)

    def _index_domains(self, dump_dir):
        """
        :type dump_dir: str|unicode
        """
        for dir_entry in os.listdir(dump_dir):
            dir_entry_path = os.path.join(dump_dir, dir_entry)
            if os.path.isdir(dir_entry_path):
                self._index_domain(dir_entry_path)

    def _index_domain(self, dump_dir):
        """
        :type dump_dir: str|unicode
        """
        for dir_entry in os.listdir(dump_dir):
            dir_entry_path = os.path.join(dump_dir, dir_entry)
            if os.path.isdir(dir_entry_path):
                continue
            if self._is_extra_dump_file(dir_entry):
                self._index_entity('domain', dir_entry_path)

    def _index_entity(self, entity_type, entity_dump_path):
        """
        :type entity_type: str|unicode
        :type entity_dump_path: str|unicode
        """
        entity_node = ElementTree.parse(entity_dump_path).find(entity_type)
        if entity_node is None:
            return
        entity_name = entity_node.get('name')
        if entity_name:
            self._extra_dump_entities[entity_type][entity_name] = entity_dump_path

    def _get_extra_dump_file(self, entity_type, entity_name):
        """
        :type entity_type: str|unicode
        :type entity_name: str|unicode
        :rtype: str|unicode|None
        """
        files = self._extra_dump_entities.get(entity_type)
        if files is not None:
            return files.get(entity_name)
        return None

    def _is_extra_dump_file(self, dump_file):
        """
        :type dump_file: str|unicode
        :return: bool
        """
        dump_file_name = os.path.basename(dump_file)
        match = re.search(r'^([^_]+).*_(\d+)\.xml$', dump_file_name)
        if match is None:
            return False
        if self._extra_dump_prefix != match.group(1) or self._extra_dump_timestamp != match.group(2):
            return False
        return True

    def _is_xml_file(self, dump_file):
        """
        :type dump_file: str|unicode
        :return: bool
        """
        return dump_file.endswith('.xml')

    def _get_xml_element_schema(self, schema_node):
        """
        :type schema_node: xml.etree.ElementTree.Element | None
        :rtype: xml.etree.ElementTree.Element | None
        """
        if schema_node is None:
            return None
        ref_node_name = schema_node.get('ref')
        if ref_node_name:
            schema_ref_node = self._schema.find("xs:element[@name='%s']" % ref_node_name, self._schema_namespaces)
            return self._get_xml_element_schema(schema_ref_node)
        type_name = schema_node.get('type', schema_node.get('base'))
        if type_name:
            element_schema_node = self._schema.find("xs:complexType[@name='%s']" % type_name, self._schema_namespaces)
            return element_schema_node
        element_schema_node = schema_node.find("xs:complexType", self._schema_namespaces)
        return element_schema_node

    def _iter_xml_element_schema_sequence(self, complex_type_schema_node):
        """
        :type complex_type_schema_node: xml.etree.ElementTree.Element
        :rtype: collections.Iterable[tuple[str | unicode, xml.etree.ElementTree.Element | None]]
        """
        for schema_node in complex_type_schema_node:
            if schema_node.tag == self._xs_types['complexContent']:
                extension_schema_node = schema_node.find("xs:extension", self._schema_namespaces)
                if extension_schema_node is not None:
                    base_schema_node = self._get_xml_element_schema(extension_schema_node)
                    if base_schema_node is not None:
                        for node_name, element_schema_node in self._iter_xml_element_schema_sequence(base_schema_node):
                            yield node_name, element_schema_node
                    for node_name, element_schema_node in self._iter_xml_element_schema_sequence(extension_schema_node):
                        yield node_name, element_schema_node
            if schema_node.tag in [self._xs_types['sequence'], self._xs_types['choice'], self._xs_types['all']]:
                for node_name, element_schema_node in self._iter_xml_element_schema_sequence(schema_node):
                    yield node_name, element_schema_node
            if schema_node.tag == self._xs_types['element']:
                node_name = schema_node.get('name')
                if node_name is None:
                    node_name = schema_node.get('ref')
                if node_name is not None:
                    element_schema_node = self._get_xml_element_schema(schema_node)
                    yield node_name, element_schema_node

    def _load_schema(self, schema_path):
        """
        :type schema_path: str | unicode
        :rtype: xml.etree.ElementTree.ElementTree
        """
        logger.debug(safe_format(messages.LOAD_SCHEMA_FROM, path=schema_path))
        schema = ElementTree.parse(schema_path)
        for include_node in schema.findall("xs:include", self._schema_namespaces):
            schema_location = include_node.get('schemaLocation')
            if schema_location:
                add_schema_path = os.path.join(os.path.dirname(schema_path), schema_location)
                add_schema = self._load_schema(add_schema_path)
                schema.getroot().extend(add_schema.findall("*", self._schema_namespaces))
        return schema

    def _find_child_xml_element(self, node, pattern_node):
        """
        :type node: xml.etree.ElementTree.Element
        :type pattern_node: xml.etree.ElementTree.Element
        :rtype: int
        """
        get_identity = self._identities.get(pattern_node.tag, lambda x: None)
        pattern_node_identity = get_identity(pattern_node)
        for child_node_index, child_node in enumerate(node):
            if child_node.tag != pattern_node.tag:
                continue
            child_node_identity = get_identity(child_node)
            if child_node_identity == pattern_node_identity:
                return child_node_index
        return -1
