from parallels.core import messages
from parallels.core.actions.base.subscription_action import SubscriptionAction
from parallels.core.actions.utils.logging_properties \
	import LoggingProperties
from parallels.core.target_panels import TargetPanels
from parallels.core.actions.hosting_settings.convert.plesk.dns \
	import DNS as ActionConvertDNSPlesk
from parallels.core.actions.hosting_settings.convert.ppa.dns \
	import DNS as ActionConvertDNSPPA


class DNS(SubscriptionAction):
	"""Convert DNS zones of subscriptions.

	During migration, DNS resource records of the domains need to be
	updated. Depending on source and target server panel types, some
	of the records should be removed, some new records added, and IP
	addresses should be pointed to new panel's IPs.

	Implementations of DNS record conversion are contained in the module
	common/converter/dns.py. When implementing this method, you may choose
	one of existing implementations, or you may need to create your own.

	Args:
		backup: migration dump in Plesk backup format
		source_panel_id: name of the panel section in config.ini
		subscription_target_services: target server IPs and database connection
		parameters to be used by subscriptions after migration
	"""

	def get_description(self):
		return "Convert DNS records"

	def get_failure_message(self, global_context, subscription):
		return messages.FAILED_CONVERT_DNS_RECORDS_FOR_SUBSCRIPTION % (subscription.name)

	def get_logging_properties(self):
		return LoggingProperties(info_log=False)

	def filter_subscription(self, global_context, subscription):
		"""
		:type global_context: parallels.core.global_context.GlobalMigrationContext
		:type subscription: parallels.core.migrated_subscription.MigratedSubscription
		"""
		return True

	def run(self, global_context, subscription):
		"""
		:type global_context: parallels.core.global_context.GlobalMigrationContext
		:type subscription: parallels.core.migrated_subscription.MigratedSubscription
		"""
		if global_context.target_panel in [TargetPanels.PLESK, TargetPanels.PVPS]:
			ActionConvertDNSPlesk().run(global_context, subscription)
		elif global_context.target_panel == TargetPanels.PPA:
			ActionConvertDNSPPA().run(global_context, subscription)
		else:
			raise NotImplementedError(
				messages.DNS_CONVERSION_IS_IMPLEMENTED_ONLY_FOR)
