from parallels.source.custom import messages
from parallels.core.utils.common import find_only


class LightBackup(object):
	def __init__(self, backup_id, data):
		self._backup_id = backup_id
		self._data = data

	def iter_all_subscriptions(self):
		"""
		:rtype: collections.Iterable[parallels.custom_panel_migrator.light_backup.LightBackupSubscription]
		"""
		for client_data in self._data:
			for subscription_data in client_data.get('subscriptions', []):
				yield LightBackupSubscription(subscription_data)

	def get_subscription(self, name):
		"""
		:type name: basestring
		:rtype: parallels.custom_panel_migrator.light_backup.LightBackupSubscription
		"""
		return find_only(
			self.iter_all_subscriptions(), lambda s: s.name == name,
			messages.FAILED_TO_FIND_SUBSCRIPTION_IN_DUMP % name
		)


class LightBackupWebFilesMappingProvider(object):
	def __init__(self, data):
		self._data = data

	def iter_web_files(self):
		for web_file_data in self._data.get('web_files', []):
			yield LightBackupWebFilesMapping(web_file_data)


class LightBackupSubscription(LightBackupWebFilesMappingProvider):
	def __init__(self, data):
		super(LightBackupSubscription, self).__init__(data)
		self._data = data

	@property
	def name(self):
		"""
		:rtype: basestring
		"""
		return self._data['name']

	@property
	def source_document_root(self):
		"""
		:rtype: basestring | dict | None
		"""
		return self._data.get('source_document_root')

	@property
	def source_webspace_root(self):
		"""
		:rtype: basestring | dict | None
		"""
		return self._data.get('source_webspace_root')

	def iter_databases(self):
		"""
		:rtype: collections.Iterable[parallels.custom_panel_migrator.light_backup.LightBackupDatabase]
		"""
		for database_data in self._data.get('databases', []):
			yield LightBackupDatabase(database_data)

	def iter_mailboxes(self):
		"""
		:rtype: collections.Iterable[parallels.source.custom.light_backup.LightBackupMailbox]
		"""
		for mailbox_data in self._data.get('mailboxes', []):
			yield LightBackupMailbox(mailbox_data)

	def iter_addon_domains(self):
		"""
		:rtype: collections.Iterable[parallels.source.custom.light_backup.LightBackupAddonDomain]
		"""
		for addon_domain_data in self._data.get('addon_domains', []):
			yield LightBackupAddonDomain(addon_domain_data)

	def iter_subdomains(self):
		"""
		:rtype: collections.Iterable[parallels.source.custom.light_backup.LightBackupSubdomain]
		"""
		for subdomain_data in self._data.get('subdomains', []):
			yield LightBackupSubdomain(subdomain_data)

	def get_addon_domain(self, name):
		"""
		:type name: basestring
		:rtype: parallels.custom_panel_migrator.light_backup.LightBackupAddonDomain
		"""
		return find_only(
			self.iter_addon_domains(), lambda d: d.name == name,
			messages.FAILED_TO_FIND_ADDON_IN_DUMP % name
		)

	def get_subdomain(self, name):
		"""
		:type name: basestring
		:rtype: parallels.custom_panel_migrator.light_backup.LightBackupSubdomain
		"""
		return find_only(
			self.iter_subdomains(), lambda d: d.name == name,
			messages.FAILED_TO_FIND_SUBDOMAIN_IN_DUMP % name
		)


class LightBackupDatabase(object):
	def __init__(self, data):
		self._data = data

	@property
	def db_type(self):
		"""
		:rtype: basestring
		"""
		return self._data.get('type', 'mysql')

	@property
	def name(self):
		"""
		:rtype: basestring
		"""
		return self._data['name']

	@property
	def dump(self):
		"""
		:rtype: basestring | None
		"""
		return self._data.get('dump')


class LightBackupMailbox(object):
	def __init__(self, data):
		self._data = data

	@property
	def name(self):
		"""
		:rtype: str
		"""
		return self._data['name']

	@property
	def short_name(self):
		"""
		:rtype: str
		"""
		return self.name[:self.name.find('@')]

	@property
	def domain_name(self):
		"""
		:rtype: str
		"""
		return self.name[self.name.find('@')+1:]

	@property
	def source_directory(self):
		"""
		:rtype: basestring | None
		"""
		return self._data.get('directory')


class LightBackupAddonDomain(LightBackupWebFilesMappingProvider):
	def __init__(self, data):
		super(LightBackupAddonDomain, self).__init__(data)
		self._data = data

	@property
	def name(self):
		"""
		:rtype: str
		"""
		return self._data['name']

	@property
	def source_document_root(self):
		"""
		:rtype: basestring | dict | None
		"""
		return self._data.get('source_document_root')


class LightBackupSubdomain(LightBackupWebFilesMappingProvider):
	def __init__(self, data):
		super(LightBackupSubdomain, self).__init__(data)
		self._data = data

	@property
	def name(self):
		"""
		:rtype: str
		"""
		return self._data['name']

	@property
	def source_document_root(self):
		"""
		:rtype: basestring | dict | None
		"""
		return self._data.get('source_document_root')


class LightBackupWebFilesMapping(object):
	def __init__(self, data):
		self._data = data

	@property
	def source(self):
		return self._data['source']

	@property
	def target(self):
		return self._data['target']

	@property
	def exclude(self):
		return self._data.get('exclude', [])

	@property
	def skip_if_source_not_exists(self):
		return self._data.get('skip_if_source_not_exists', False)
