from parallels.source.custom import messages
import logging
from contextlib import closing

import connections

from parallels.core.migrator import Migrator as CommonMigrator
from parallels.source.custom.backup_creator import BackupCreator
from parallels.source.custom.migrated_subscription import CustomPanelMigratedSubscription
from parallels.source.custom.session_files import CustomPanelSessionFiles
from parallels.source.custom.workflow import FromCustomPanelWorkflow
from parallels.source.custom.global_context import CustomPanelGlobalMigrationContext

logger = logging.getLogger(__name__)


class Migrator(CommonMigrator):
	logger = logging.getLogger(__name__)

	def __init__(self, config):
		super(Migrator, self).__init__(config)
		self.global_context.source_plesks = None

	def iter_shallow_plesk_backups(self):
		return self.iter_plesk_backups()

	def iter_plesk_backups(self):
		for source_info in self.global_context.get_sources_info():
			yield source_info.id, source_info.load_raw_backup()

	def iter_converted_plesk_backups(self):
		for source_info in self.global_context.get_sources_info():
			with closing(self.load_converted_plesk_backup(source_info.id)) as backup:
				yield source_info.id, backup

	def iter_converted_backups(self):
		return self.iter_converted_plesk_backups()

	def create_full_backup_from_light_backup(self):
		backup_creator = BackupCreator()
		backup_creator.write_full_backup(
			light_backup_config=connections.LightBackupConfig(
				source_id='light-backup',
				path=self.global_context.options.source_file,
				file_format=self.global_context.options.source_format
			),
			target_backup_filename=self.global_context.options.target_file,
			database_servers=list(self.global_context.conn.iter_database_servers())
		)

	def _subscription_is_windows(self, subscription_name, plesk_id=None):
		if plesk_id is not None:
			return self.get_raw_plesk_backup(plesk_id).is_windows
		for _, backup in self.iter_plesk_backups():
			if backup.has_subscription(subscription_name):
				return backup.is_windows
		raise Exception(messages.UNABLE_TO_GET_BACKUP_WITH_SPECIFIED_SUBSCRIPTION)

	def _get_source_servers(self):
		return {}

	def _load_connections_configuration(self):
		return connections.MigratorConnections(self.config, self._get_target_panel())

	def _create_workflow(self):
		return FromCustomPanelWorkflow()

	def _create_global_context(self):
		return CustomPanelGlobalMigrationContext()

	def _create_session_files(self):
		return CustomPanelSessionFiles(self.conn, self._get_migrator_server())

	def _get_source_web_node(self, subscription_name):
		return self._get_source_subscription_node(subscription_name)

	def _get_source_mail_node(self, subscription_name):
		return self._get_source_subscription_node(subscription_name)

	def _get_source_subscription_node(self, subscription_name):
		subscription = self._create_migrated_subscription(subscription_name)
		source_node_id = subscription.model.source
		return self.global_context.conn.get_source_server_by_id(source_node_id)

	def _get_source_db_node(self, source_id):
		return self.global_context.conn.get_source_server_by_id(source_id)

	def _create_migrated_subscription(self, name):
		return CustomPanelMigratedSubscription(self, name)
