import posixpath

from parallels.core.actions.content.web.copy_unix_content_base \
	import CopyUnixWebContentBase, CopyWebContentItem
from parallels.core.utils import plesk_utils
from parallels.core.utils.paths import web_paths


class CopyUnixWebContent(CopyUnixWebContentBase):
	def _list_files_to_copy(self, global_context, subscription):
		"""Make a list of source server directories to be transferred.

		Return a list of (source directory -> destination directory) mappings

		:rtype: list[parallels.core.actions.content.web.copy_unix_content_base.CopyWebContentItem]
		"""
		vhost_name = subscription.name_idn
		source_vhosts_dir = subscription.web_source_server.vhosts_dir

		# chroot environment must be recreated, not copied.
		# That is necessary because binaries and libraries from old system
		# may be incompatible with target server (for example, other OS or architecture),
		# so customer won't be able to login to the system after moving.
		chroot_excludes = [
			'/%s' % name for name in ('bin', 'dev', 'lib', 'lib64', 'usr', 'libexec', 'etc', 'tmp', 'var')
		]

		if subscription.web_source_server.plesk_version >= (11, 5):
			# since 11.5 we have a new virtual host structure: each domain is
			# splitted over VHOSTS_D/<domain_name> and
			# VHOSTS_D/system/<domain_name>
			tocopy = [
				CopyWebContentItem(
					source_path=web_paths.AbsolutePath(posixpath.join(source_vhosts_dir, vhost_name)),
					target_path=web_paths.WebspaceRoot(subscription.converted_backup),
					exclude=[
						'/logs', '/conf', '/statistics',
					] + chroot_excludes,
				),
				CopyWebContentItem(
					source_path=web_paths.AbsolutePath(
						posixpath.join(source_vhosts_dir, "system", vhost_name, "logs")
					),
					target_path=web_paths.WebspaceLogs(subscription.converted_backup),
					skip_if_source_not_exists=True
				),
				CopyWebContentItem(
					source_path=web_paths.AbsolutePath(
						posixpath.join(source_vhosts_dir, "system", vhost_name, "statistics")
					),
					target_path=web_paths.WebspaceStatistics(subscription.converted_backup),
					skip_if_source_not_exists=True
				),
				CopyWebContentItem(
					source_path=web_paths.AbsolutePath(
						posixpath.join(source_vhosts_dir, "system", vhost_name, "pd")
					),
					target_path=web_paths.WebspaceProtectedDirs(subscription.converted_backup),
					skip_if_source_not_exists=True
				),
			]
		else:
			tocopy = [
				CopyWebContentItem(
					source_path=web_paths.AbsolutePath(posixpath.join(source_vhosts_dir, vhost_name)),
					target_path=web_paths.WebspaceRoot(subscription.converted_backup),
					exclude=["/logs", "/conf", "/statistics", "/pd"] + chroot_excludes,
				),
				CopyWebContentItem(
					source_path=web_paths.AbsolutePath(
						posixpath.join(source_vhosts_dir, vhost_name, "statistics", "logs")
					),
					target_path=web_paths.WebspaceLogs(subscription.converted_backup),
					skip_if_source_not_exists=True
				),
				CopyWebContentItem(
					source_path=web_paths.AbsolutePath(
						posixpath.join(source_vhosts_dir, vhost_name, "statistics")
					),
					target_path=web_paths.WebspaceStatistics(subscription.converted_backup),
					exclude=['/logs'],
					skip_if_source_not_exists=True
				),
				CopyWebContentItem(
					source_path=web_paths.AbsolutePath(
						posixpath.join(source_vhosts_dir, vhost_name, "pd")
					),
					target_path=web_paths.WebspaceProtectedDirs(subscription.converted_backup),
					skip_if_source_not_exists=True
				),
			]

		for site in subscription.converted_backup.iter_sites():
			# skip sites w/o hosting
			if not site.is_virtual_hosting:
				continue

			site_vhost_name = plesk_utils.convert_wildcard_to_path(site.name.encode('idna'))
			
			if subscription.web_source_server.plesk_version >= (10, 0):  # plesk #10.x
				if subscription.web_source_server.plesk_version >= (11, 5):
					tocopy.append(
						CopyWebContentItem(
							source_path=web_paths.AbsolutePath(
								posixpath.join(source_vhosts_dir, "system", site_vhost_name, "pd")
							),
							target_path=web_paths.SiteProtectedDirs(subscription.converted_backup, site),
							skip_if_source_not_exists=True
						)
					)

					source_logs_subdir = posixpath.join(
						source_vhosts_dir, "system",
						site_vhost_name, "logs"
					)
					source_statistics_subdir = posixpath.join(
						source_vhosts_dir, 'system',
						site_vhost_name, "statistics"
					)
					source_statistics_exclude = []
				else:
					tocopy.append(
						CopyWebContentItem(
							source_path=web_paths.AbsolutePath(
								posixpath.join(source_vhosts_dir, site_vhost_name, "pd")
							),
							target_path=web_paths.SiteProtectedDirs(subscription.converted_backup, site),
							skip_if_source_not_exists=True
						)
					)

					source_logs_subdir = posixpath.join(
						source_vhosts_dir, site_vhost_name,
						"statistics", "logs"
					)
					source_statistics_subdir = posixpath.join(
						source_vhosts_dir, site_vhost_name,
						"statistics"
					)
					source_statistics_exclude = ["/logs"]
				
				tocopy.extend([
					# site logs
					CopyWebContentItem(
						source_path=web_paths.AbsolutePath(source_logs_subdir),
						target_path=web_paths.SiteLogs(subscription.converted_backup, site),
						skip_if_source_not_exists=True
					),
					# site statistics
					CopyWebContentItem(
						source_path=web_paths.AbsolutePath(source_statistics_subdir),
						target_path=web_paths.SiteStatistics(subscription.converted_backup, site),
						exclude=source_statistics_exclude,
						skip_if_source_not_exists=True
					),
				])
			elif subscription.web_source_server.plesk_version >= (8, 0):  # plesk #8.x and #9.x
				pass

		return tocopy