import sys
import logging

from Queue import Queue
from threading import Thread

from plesk_mail_migrator.core.dumps.queue_reader import QueueDumpReader
from plesk_mail_migrator.core.dumps.queue_writer import QueueDumpWriter
from plesk_mail_migrator.core.dumps.stream_reader import StreamDumpReader
from plesk_mail_migrator.core.dumps.stream_writer import StreamDumpWriter
from plesk_mail_migrator.core.entities.mail_account import MailAccount
from plesk_mail_migrator.core.entities.utils.imap_flags_utils import IMAPFlagsUtils
from plesk_mail_migrator.providers.imap.provider import IMAPProvider
from plesk_mail_migrator.providers.maildir.provider import MaildirProvider
from plesk_mail_migrator.providers.provider_factory import ProviderFactory
from plesk_mail_migrator.utils.cmd_args.plain_cmd_args_parser import PlainCmdArgsParser
from plesk_mail_migrator.utils.cmd_args.cmd_args_parser import CommandLineArgumentException
from plesk_mail_migrator.utils.cmd_args.prefixed_cmd_args_parser import PrefixedCmdArgsParser
from plesk_mail_migrator.utils.exceptions import MailMigratorException
from plesk_mail_migrator.utils.file_utils import open_wrap_not_exists

logging.getLogger('plesk_mail_migrator').addHandler(logging.NullHandler())
logger = logging.getLogger(__name__)


DEFAULT_QUEUE_SIZE = 100


def main():
    try:
        if len(sys.argv) > 1:
            command_name = sys.argv[1]
            argparser = PlainCmdArgsParser(sys.argv[2:])
            configure_logger(argparser)

            if command_name == 'backup':
                logger.info("START Backup")
                mail_account = create_mail_account(argparser)
                provider = create_backup_provider(argparser)
                exclude_message_ids = read_message_ids(argparser)
                argparser.check_unused_arguments()
                command_backup(mail_account, provider, exclude_message_ids)
                logger.info("END Backup")
            elif command_name == 'restore':
                logger.info("START Restore")
                mail_account = create_mail_account(argparser)
                provider = create_restore_provider(argparser)
                argparser.check_unused_arguments()
                command_restore(mail_account, provider)
                logger.info("END Restore")
            elif command_name == 'list-message-ids':
                logger.info("START List message IDs")
                mail_account = create_mail_account(argparser)
                provider = create_restore_provider(argparser)
                filename = argparser.get('output-file')
                argparser.check_unused_arguments()
                command_list_message_ids(mail_account, provider, filename)
                logger.info("END List message IDs")
            elif command_name == 'migrate':
                logger.info("START Migrate")
                source_argparser = PrefixedCmdArgsParser(argparser, prefix="source")
                target_argparser = PrefixedCmdArgsParser(argparser, prefix="target")
                source_mail_account = create_mail_account(source_argparser, "Source mail account")
                target_mail_account = create_mail_account(target_argparser, "Target mail account")
                source_provider = create_backup_provider(source_argparser)
                target_provider = create_restore_provider(target_argparser)
                if argparser.contains('queue-size'):
                    queue_size = argparser.get('queue-size')
                else:
                    queue_size = DEFAULT_QUEUE_SIZE
                argparser.check_unused_arguments()
                command_migrate(
                    source_mail_account, source_provider, target_mail_account, target_provider, queue_size
                )
                logger.info("END Migrate")
            elif command_name == 'print-dump-contents':
                logger.info("START Print dump contents")
                if not argparser.contains("dump-file"):
                    raise CommandLineArgumentException(
                        "Dump file is not specified. Specify it with --dump-file option."
                    )
                dump_file = argparser.get('dump-file')
                argparser.check_unused_arguments()
                command_print_dump_contents(dump_file)
                logger.info("END Print dump contents")
            elif command_name in ['help', '--help', '/help', '-h', '-?', '/h', '/?']:
                logger.info("START Help")
                command_help()
                logger.info("END Help")
            else:
                raise CommandLineArgumentException(
                    "Invalid usage: invalid command was specified.\n"
                    "Allowed commands: backup, restore, list-message-ids, print-dump-contents, help."
                )
        else:
            raise CommandLineArgumentException("Invalid usage: no commands were specified")
    except CommandLineArgumentException as e:
        logger.debug("Exception: ", exc_info=True)
        logger.error("Command line arguments exception: %s", str(e))
        sys.stderr.write("%s\n" % (str(e),))
        sys.exit(1)
    except MailMigratorException as e:
        logger.debug("Exception: ", exc_info=True)
        logger.error(str(e))
        sys.stderr.write("%s\n" % (str(e),))
        sys.exit(1)
    except Exception as e:
        logger.debug("Exception: ", exc_info=True)
        logger.debug("Internal error of mail migrator: %s", str(e))
        sys.stderr.write("Internal error of mail migrator: %s\n" % (str(e),))
        sys.exit(1)


def command_backup(account, provider, exclude_message_ids):
    dumper = StreamDumpWriter()
    errors = provider.do_backup_messages(account, dumper, exclude_message_ids)
    if len(errors) > 0:
        raise MailMigratorException("\n\n".join(errors))


def command_restore(account, provider):
    dump_reader = StreamDumpReader()
    while True:
        message = dump_reader.read_message()
        if message is None:
            break

        provider.do_restore_message(account, message)


def command_migrate(source_mail_account, source_provider, target_mail_account, target_provider, queue_size):
    logger.debug("Queue size: %s", queue_size)
    queue = Queue(queue_size)
    dump_reader = QueueDumpReader(queue)
    dump_writer = QueueDumpWriter(queue)

    existing_message_ids = target_provider.list_message_ids(target_mail_account)
    errors = []

    def restore_thread_main():
        try:
            message_num = 1
            while True:
                message = dump_reader.read_message()
                if message is None:
                    break
                try:
                    target_provider.do_restore_message(target_mail_account, message)
                except Exception as e:
                    logger.debug("Exception: ", exc_info=True)
                    errors.append("Failed to restore mail message #%s: %s" % (message_num, str(e)))
                message_num += 1
        except Exception as e:
            logger.debug("Exception: ", exc_info=True)
            errors.append("Failed to restore mail messages: %s" % str(e))

    def backup_thread_main():
        try:
            errors.extend(
                source_provider.do_backup_messages(source_mail_account, dump_writer, existing_message_ids)
            )
        except Exception as e:
            logger.debug("Exception: ", exc_info=True)
            errors.append("Failed to backup mail messages: %s" % str(e))
        finally:
            # put "None" marker meaning that there are no more messages
            queue.put(None)

    backup_thread = Thread(target=backup_thread_main, name='BackupThread')
    restore_thread = Thread(target=restore_thread_main, name='RestoreThread')
    backup_thread.start()
    restore_thread.start()
    backup_thread.join()
    restore_thread.join()

    if len(errors) > 0:
        raise MailMigratorException("\n\n".join(errors))


def command_list_message_ids(account, provider, filename):
    message_ids = provider.list_message_ids(account)
    with open(filename, 'w') as fp:
        fp.write("\n".join(message_ids))
        fp.write("\n")


def command_print_dump_contents(filename):
    not_exists_error_message = (
        "Dump file '{file}' with mail messages does not exist. Specify correct path to file."
    ).format(
        file=filename
    )
    with open_wrap_not_exists(filename, not_exists_error_message) as fp:
        dump_reader = StreamDumpReader(fp)

        first = True
        messages_count = 0

        while True:
            message = dump_reader.read_message()
            if message is None:
                break

            if not first:
                print
                print
            else:
                first = False

            print "=============MESSAGE BEGIN==================="
            print "Number: %s" % (messages_count + 1)
            print "Message ID: %s" % message.message_id
            print "IMAP Folder: %s" % message.folder
            print "IMAP Flags: %s" % IMAPFlagsUtils.flags_to_string(message.flags)
            print "Contents:"
            print "*********************************************"
            print message.body
            print "===============MESSAGE END==================="

            messages_count += 1

        if not first:
            print
            print

        print "Total messages: %s" % messages_count


def command_help():
    help_message = """Tool used to migrate mail messages between mail accounts.
Usage:
mail-migrator <command> <arguments>

Available commands:
- backup - backup messages of mail account
- restore - restore messages of mail account
- list-message-ids - list message IDs of mail account
- print-dump-contents - print dump contents in human readable format
- help - display that help message

"backup" command
The command writes dump of account messages to standard output.
The dump has binary format.
Arguments:
--provider=<provider-id> - specify mail provider (mail server)
If no mail provider was specified, the tool will use IMAP mail provider.
--domain=<domain> - specify domain name of mail account
(everything after '@' symbol)
--mailname=<mailname> - specify mail name of mail account
(everything before '@' symbol)
--password=<password> - specify password of mail account
--account-description-file=<filename> - filename describing mail account to backup
File should contain domain name, mail name and password delimited by newlines.
Either accounts description file, or all domain name, mail name and password options should be specified.
Use the file to avoid password in command line.
--exclude-message-ids-file=<filename> - file with message ids that should not be dumped,
generated by list-message-ids command

"restore" command
The command expects dump (generated by "backup" command) at standard input
Arguments:
--provider=<provider-id> - specify mail provider (mail server)
If no mail provider was specified, the tool will use mail directory provider.
--domain=<domain> - specify domain name of mail account
(everything after '@' symbol)
--mailname=<mailname> - specify mail name of mail account
(everything before '@' symbol)
--password=<password> - specify password of mail account
--account-description-file=<filename> - filename describing mail account to backup
File should contain domain name, mail name and password delimited by newlines.
Either accounts description file, or all domain name, mail name and password options should be specified.

"list-message-ids" command
List message IDs that are already exist on the server.
That command should be executed on the target server, and
its results should be provided to the source server
with --exclude-message-ids-file option
Arguments:
--provider=<provider-id> - specify mail provider (mail server)
If no mail provider was specified, the tool will use mail directory provider.
--domain=<domain> - specify domain name of mail account
(everything after '@' symbol)
--mailname=<mailname> - specify mail name of mail account
(everything before '@' symbol)
--password=<password> - specify password of mail account
--account-description-file=<filename> - filename describing mail account to backup
File should contain domain name, mail name and password delimited by newlines.
Either accounts description file, or all domain name, mail name and password options should be specified.
--output-file=<filename> - specify filename to write message IDs to

"migrate" command
Migrate mail messages from one mail server to another one.
The primary goal is to migrate to the server where the tool is running from another server, getting messages by IMAP.
Arguments:
--source-provider=<provider-id> - specify source mail provider (mail server)
IMAP is the only recommended option, it is selected by default.
Attention: If you specify any other mail provider than IMAP,
data will be always taken from the local server, that is not what you usually expect.
For IMAP provider specify host and port with --host and --port options.
--source-domain=<domain> - specify domain name of mail account
(everything after '@' symbol) on the source server
--source-mailname=<mailname> - specify mail name of mail account
(everything before '@' symbol) on the source server
--source-password=<password> - specify password of mail account on the source server
--source-account-description-file=<filename> - filename describing source mail account to migrate messages from
File should contain domain name, mail name and password delimited by newlines.
Either accounts description file, or all domain name, mail name and password options
should be specified for the source server.
--target-provider=<provider-id> - specify target mail provider (mail server).
If not specified, mail migrator will use mail directory provider.
--target-domain=<domain> - specify domain name of mail account
(everything after '@' symbol) on the target server
--target-mailname=<mailname> - specify mail name of mail account
(everything before '@' symbol) on the target server
--target-password=<password> - specify password of mail account on the target server
--target-account-description-file=<filename> - filename describing target mail account to migrate messages to
File should contain domain name, mail name and password delimited by newlines.
Either accounts description file, or all domain name, mail name and password options
should be specified for the target server.
--queue-size=<queue-size> - size of migration queue - maximum number of migrated messages that are stored in memory.
Increasing that parameter may improve speed of migration, but will require more memory.

Provider-specific arguments for "migrate" command should be prefixed by "--source-" or "--target-" strings,
for source and target servers accordingly. For example, if provider has option "host", then
to specify source host "example.com" you should specify that option as "--source-host=example.com"

"print-dump-contents" command
Use this command for debugging, when you want to view
contents of binary dump generated by "backup" command.
Arguments:
--dump-file=<filename> - specify name of a file with dump
generated by "backup" command

Overall arguments, applicable to all commands:
--log-file=<filename> - specify name of a file to write log with debugging information to"""
    print help_message

    print
    print "Available providers for backup:"
    for provider in ProviderFactory.get_backup_providers():
        print '- %s (%s)' % (provider.get_provider_id(), provider.get_title())

    print
    print "Available providers for restore:"
    for provider in ProviderFactory.get_restore_providers():
        print '- %s (%s)' % (provider.get_provider_id(), provider.get_title())

    print
    print 'Additional arguments that could be passed for specific providers'
    print '(add "source" or "target" prefix in case of "migrate" command):'

    for provider in ProviderFactory.get_all_providers():
        parameters = provider.get_parameters()
        if len(parameters) > 0:
            print "Provider '%s'" % provider.get_provider_id()
            for parameter in parameters:
                print "--%s=<%s>: %s" % (parameter.name, parameter.help_value, parameter.description)


def create_mail_account(argparser, log_mail_account_title=None):
    """Get information about account to backup/restore/migrate

    :type log_mail_account_title: str | None
    :type argparser: plesk_mail_migrator.utils.cmd_args.cmd_args_parser.CmdArgsParser
    :rtype: plesk_mail_migrator.core.entities.mail_account.MailAccount
    """
    if log_mail_account_title is None:
        log_mail_account_title = "Mail account"

    if argparser.contains('account-description-file'):
        description_file_name = argparser.get('account-description-file')
        not_exists_error_message = (
            "File '{file}' with account description does not exist. Specify correct path to file."
        ).format(
            file=description_file_name
        )
        with open_wrap_not_exists(description_file_name, not_exists_error_message) as fp:
            description_file_contents = fp.read()
        description_file_lines = description_file_contents.strip().split("\n")

        if len(description_file_lines) != 3:
            raise CommandLineArgumentException(
                "Account description file '{file}' has invalid format. "
                "It should have 3 lines, while it actually has {actual_lines_count} lines. "
                "The first line should contain domain name, the second line should contain mailname and "
                "the third line should contain password.".format(
                    file=description_file_name,
                    actual_lines_count=len(description_file_lines)
                )
            )

        domain_name = description_file_lines[0].strip()
        mailname = description_file_lines[1].strip()
        password = description_file_lines[2].strip()
    else:
        if (
            not argparser.contains('domain') or
            not argparser.contains('mailname') or
            not argparser.contains('password')
        ):
            raise CommandLineArgumentException(
                "Information about account was not specified. Please specify it by providing "
                "{domain_param_name}, {mailname_param_name} and {password_param_name} arguments "
                "(all of them must be specified). "
                "Alternatively you could provide this information with file using "
                "{account_description_file_param_name} option. The file should contain domain name, "
                "mailname and password delimited by newline character. "
                "Use the file to avoid plain text password in command line. ".format(
                    domain_param_name=argparser.get_full_param_name('domain'),
                    mailname_param_name=argparser.get_full_param_name('mailname'),
                    password_param_name=argparser.get_full_param_name('password'),
                    account_description_file_param_name=argparser.get_full_param_name('account-description-file')
                )
            )
        else:
            domain_name = argparser.get('domain')
            mailname = argparser.get('mailname')
            password = argparser.get('password')

    logger.info("%s: %s@%s", log_mail_account_title, mailname, domain_name)
    return MailAccount(mailname, password, domain_name)


def create_backup_provider(argparser):
    """Create backup provider, according to provided command-line options

    :type argparser: plesk_mail_migrator.utils.cmd_args.cmd_args_parser.CmdArgsParser
    """
    providers = {
        'imap': IMAPProvider()
    }

    default_provider_id = 'imap'

    if argparser.contains('provider'):
        provider_id = argparser.get('provider')
    else:
        provider_id = default_provider_id

    if provider_id not in providers:
        raise CommandLineArgumentException(
            "Invalid backup provider was specified by '{option}' option: {specified}. "
            "Allowed values are: {allowed}".format(
                option=argparser.get_full_param_name('provider'),
                specified=provider_id, allowed=", ".join(providers.keys())
            )
        )
    provider = providers[provider_id]

    _fill_provider_parameters(provider, argparser)
    logger.info("Backup provider: %s", provider.get_title())
    for parameter in provider.get_parameters():
        logger.info("Backup provider parameter: %s=%s", parameter.name, parameter.value)

    return provider


def create_restore_provider(argparser):
    """Create restore provider, according to provided command-line options

    :type argparser: plesk_mail_migrator.utils.cmd_args.cmd_args_parser.CmdArgsParser
    """
    providers = {
        'maildir': MaildirProvider()
    }
    default_provider_id = 'maildir'

    if argparser.contains('provider'):
        provider_id = argparser.get('provider')
    else:
        provider_id = default_provider_id

    if provider_id not in providers:
        raise CommandLineArgumentException(
            "Invalid restore provider was specified by '{option}' option: {specified}. "
            "Allowed values are: {allowed}".format(
                option=argparser.get_full_param_name('provider'),
                specified=provider_id, allowed=", ".join(providers.keys())
            )
        )
    provider = providers[provider_id]

    _fill_provider_parameters(provider, argparser)
    logger.info("Restore provider: %s", provider.get_title())
    for parameter in provider.get_parameters():
        logger.info("Restore provider parameter: %s=%s", parameter.name, parameter.value)
    return provider


def _fill_provider_parameters(provider, argparser):
    """Fill provider parameters from command-line

    :type provider: plesk_mail_migrator.core.provider.base_provider.BaseProvider
    :type argparser: plesk_mail_migrator.utils.cmd_args_parser.CmdArgsParser
    """
    for parameter in provider.get_parameters():
        if argparser.contains(parameter.name):
            parameter.value = argparser.get(parameter.name)


def read_message_ids(argparser):
    """Read message IDs to exclude from migration from file specified as command line argument.

    :type argparser: plesk_mail_migrator.utils.cmd_args_parser.CmdArgsParser
    :rtype: set[str]
    """
    message_ids = set()

    if argparser.contains("exclude-message-ids-file"):
        message_ids_file = argparser.get("exclude-message-ids-file")
        not_exists_error_message = (
            "File '{file}' with excluded messages IDs does not exist. Specify correct path to file."
        ).format(
            file=message_ids_file
        )
        with open_wrap_not_exists(message_ids_file, not_exists_error_message) as fp:
            for line in fp.readlines():
                line = line.strip()
                if line != '':
                    message_ids.add(line)
        logger.info("Exclude message IDs file '%s' contains %s IDs", message_ids_file, len(message_ids))

    return message_ids


def configure_logger(argparser):
    """Configure logger according to command-line arguments

    :type argparser: plesk_mail_migrator.utils.cmd_args_parser.CmdArgsParser
    :rtype: None
    """
    if argparser.contains('log-file'):
        logging.basicConfig(
            format="%(asctime)s|%(levelname)s|%(message)s",
            datefmt='%Y-%m-%d_%H:%M:%S,%03d',
            filename=argparser.get('log-file'),
            level=logging.DEBUG
        )


if __name__ == '__main__':
    main()
