import logging
from imaplib import IMAP4
import re

from plesk_mail_migrator.core.entities.imap_flags import IMAPFlags
from plesk_mail_migrator.core.entities.utils.imap_flags_utils import IMAPFlagsUtils
from plesk_mail_migrator.core.entities.utils.mail_message_folder_utils import MailMessageFolderUtils
from plesk_mail_migrator.core.entities.utils.mail_message_utils import MailMessageUtils
from plesk_mail_migrator.core.provider.backup_provider import BackupProvider
from plesk_mail_migrator.core.provider.provider_parameter import ProviderParameter
from plesk_mail_migrator.utils.list_utils import get_last
from plesk_mail_migrator.utils.string_utils import split_quoted_string

logger = logging.getLogger(__name__)


class IMAPProvider(BackupProvider):
    IMAP_HOST_PARAMETER = "host"
    IMAP_PORT_PARAMETER = "port"
    IMAP_FOLDER_SEPARATOR = "folder-separator"

    DEFAULT_FOLDER_SEPARATOR = '.'
    SMARTER_MAIL_FOLDER_SEPARATOR = '/'

    def __init__(self):
        self._parameters = [
            ProviderParameter(self.IMAP_HOST_PARAMETER, "IMAP host", None),
            ProviderParameter(self.IMAP_PORT_PARAMETER, "IMAP port", 143),
            ProviderParameter(
                self.IMAP_FOLDER_SEPARATOR,
                "IMAP folder separator, automatically detected if not specified",
                None, "separator"
            )
        ]

    def get_parameters(self):
        return self._parameters

    def get_title(self):
        return 'IMAP'

    def get_provider_id(self):
        return 'imap'

    def do_backup_messages(self, account, dumper, exclude_message_ids):
        """Dump (backup) messages of specified mail account.

        :param plesk_mail_migrator.core.entities.mail_account.MailAccount account:
            Mail account for which we should dump messages
        :param plesk_mail_migrator.core.dumps.dump_writer.DumpWriter dumper:
            Object capable of writing dumps
        :param list[str] exclude_message_ids:
            Messages to exclude when dumping. Check "How the tool works with message IDs" in README for details
        :rtype: list[str]
        """
        errors = []
        self._login(account)
        try:
            errors.extend(self._backup_messages(dumper, exclude_message_ids))
            return errors
        finally:
            self._logout(account)

    def _backup_messages(self, dumper, exclude_message_ids):
        """Dump (backup) messages, considering we are logged into IMAP server.

        :type dumper: plesk_mail_migrator.core.dumps.dump_writer.DumpWriter
        :type exclude_message_ids: list[str]
        :rtype: list[str]
        """
        errors = []

        for imap_dir_info_str in self._imap.list()[1]:
            imap_dir_info = split_quoted_string(imap_dir_info_str)
            imap_dir_name = get_last(imap_dir_info)
            logger.debug("Dump IMAP folder '%s'", imap_dir_name)
            try:
                folder = MailMessageFolderUtils.pack(imap_dir_name, self._folder_separator)

                select_result = self._imap.select(imap_dir_name, readonly=True)[1][0]
                try:
                    messages_count = int(select_result)
                except ValueError:
                    logger.debug("Skip dumping IMAP folder: %s", select_result)
                    continue

                logger.debug("IMAP folder contains %s messages", messages_count)
                for i in range(0, messages_count):
                    try:
                        logger.debug("Dump mail message #%s in the IMAP folder", i + 1)
                        fetch_response = self._imap.fetch(i + 1, "(FLAGS RFC822)")
                        message_content = fetch_response[1][0][1]
                        flags_data = fetch_response[1][0][0]
                        flags_match = re.search(r'FLAGS\s*\((.*?)\)', flags_data)
                        if flags_match:
                            flags = IMAPFlagsUtils.flags_from_string(flags_match.group(1))
                        else:
                            flags = IMAPFlags()

                        message_id = MailMessageUtils.get_normalized_message_id(message_content)
                        logger.debug("Message ID: %s", message_id)

                        if message_id in exclude_message_ids:
                            logger.debug("Message is skipped as it is in exclude message IDs list")
                            continue

                        logger.debug("Dump message")
                        dumper.dump_from_contents(message_id, folder, flags, message_content)
                    except Exception as e:
                        logger.debug("Exception: ", exc_info=True)
                        errors.append(
                            "Failed to backup message #%s in IMAP folder '%s': %s" % (
                                i + 1, imap_dir_name, str(e)
                            )
                        )
            except Exception as e:
                logger.debug("Exception: ", exc_info=True)
                errors.append(
                    "Failed to backup IMAP folder '%s': %s. All messages of that folder will not be migrated" % (
                        imap_dir_name, str(e)
                    )
                )

        return errors

    def _login(self, account):
        """Login to the server by IMAP.

        :type account: plesk_mail_migrator.core.entities.mail_account.MailAccount
        :rtype: None
        """
        logger.debug("Login by IMAP to account %s@%s", account.user_name, account.domain)
        self._imap = IMAP4(
            self.get_parameter_value(self.IMAP_HOST_PARAMETER),
            self.get_parameter_value(self.IMAP_PORT_PARAMETER)
        )
        self._imap.login('%s@%s' % (account.user_name, account.domain), account.user_password)

        logger.debug("Detect IMAP folder separator with namespace command")

        self._folder_separator = self.get_parameter_value(self.IMAP_FOLDER_SEPARATOR)

        if self._folder_separator is not None:
            logger.debug("Use IMAP folder separator specified by argument: %s", self._folder_separator)
        else:
            logger.debug("Try to detect IMAP folder separator with NAMESPACE IMAP command")
            error_reason = ''
            try:
                namespace_info = self._imap.namespace()
                namespace_match = re.match(r'^\(\("[^)]*" "([^)]*)"\)\)', namespace_info[1][0])

                if namespace_match:
                    self._folder_separator = namespace_match.group(1)
                    logger.debug("Detected IMAP folder separator by namespace command: %s", self._folder_separator)
                else:
                    error_reason = 'have not found separator in reply to NAMESPACE command'
            except IMAP4.error as e:
                error_reason = str(e)

            if self._folder_separator is None:
                if self._imap.welcome is not None and 'smartermail' in self._imap.welcome.lower():
                    self._folder_separator = self.SMARTER_MAIL_FOLDER_SEPARATOR
                    logger.debug(
                        "Failed to detect IMAP folder separator from the server: %s. "
                        "As it looks like SmarterMail, use default separator of SmarterMail: %s",
                        error_reason,
                        self._folder_separator
                    )
                else:
                    self._folder_separator = self.DEFAULT_FOLDER_SEPARATOR
                    logger.debug(
                        "Failed to detect IMAP folder separator from the server: %s. "
                        "Use default IMAP folder separator: %s",
                        error_reason,
                        self._folder_separator
                    )

    def _logout(self, account):
        """Logout from IMAP server.

        :type account: plesk_mail_migrator.core.entities.mail_account.MailAccount
        :rtype: None
        """
        logger.debug("Logout account %s@%s by IMAP", account.user_name, account.domain)
        self._imap.logout()
