import logging
import posixpath
import os

import errno
import pwd
import grp

from plesk_mail_migrator.core.entities.utils.imap_flags_utils import IMAPFlagsUtils
from plesk_mail_migrator.core.entities.utils.mail_message_folder_utils import MailMessageFolderUtils
from plesk_mail_migrator.core.provider.provider_parameter import ProviderParameter
from plesk_mail_migrator.core.provider.restore_provider import RestoreProvider
from plesk_mail_migrator.utils.string_utils import is_empty


logger = logging.getLogger(__name__)


class MaildirProvider(RestoreProvider):
    DIR_CUR = 'cur'
    DIR_NEW = 'new'
    DIR_TMP = 'tmp'

    MAILNAMES_DIR = 'mailnames-dir'
    MAILDIR_USER = 'maildir-user'
    MAILDIR_GROUP = 'maildir-group'

    def __init__(self):
        self._parameters = [
            ProviderParameter(self.MAILNAMES_DIR, "Base directory with mail messages", '/var/qmail/mailnames/', "dir"),
            ProviderParameter(self.MAILDIR_USER, "User which should own mail messages", "popuser", "user"),
            ProviderParameter(self.MAILDIR_GROUP, "Group which should own mail messages", "popuser", "group"),
        ]

    def get_parameters(self):
        return self._parameters

    def get_title(self):
        return 'Maildir'

    def get_provider_id(self):
        return 'maildir'

    def list_message_ids(self, account):
        """List IDs of messages that were already migrated (already exist on the target server).
        That list is used on the source server to avoid backing up and transferring messages that
        were already transferred.

        See "How the tool works with message IDs" in README for more details.

        :param plesk_mail_migrator.core.entities.mail_account.MailAccount account:
            Account for which we should list existing message IDs
        :rtype: list[str]
        """
        message_ids = []

        def add_message_ids(cur_dir):
            if os.path.exists(cur_dir):
                for filename in os.listdir(cur_dir):
                    if ':' in filename:
                        message_ids.append(filename[:filename.find(':')])
                    else:
                        message_ids.append(filename)

        logger.debug("List message IDs in main folder")
        maildir_path = self._get_maildir_path(account)
        main_cur_dir = posixpath.join(maildir_path, self.DIR_CUR)
        add_message_ids(main_cur_dir)

        for folder in self._get_folders(account):
            logger.debug("List message IDs in IMAP folder '%s'", folder)
            folder_cur_dir = posixpath.join(maildir_path, '.%s' % folder, self.DIR_CUR)
            add_message_ids(folder_cur_dir)

        return message_ids

    def do_restore_message(self, account, message):
        """Restore specified mail message for specified mail account.

        :param plesk_mail_migrator.core.entities.mail_account.MailAccount account:
            Account to which message should be restored
        :param plesk_mail_migrator.core.entities.mail_message.MailMessage message:
            Mail message that should be restored
        :rtype: None
        """
        uid = pwd.getpwnam(self.get_parameter_value(self.MAILDIR_USER)).pw_uid
        gid = grp.getgrnam(self.get_parameter_value(self.MAILDIR_GROUP)).gr_gid

        maildir_path = self._get_maildir_path(account)

        message_folder = MailMessageFolderUtils.unpack(message.folder, '.')

        if is_empty(message_folder) or message_folder.lower() == 'inbox':
            message_dir_path = posixpath.join(maildir_path, 'cur')
        else:
            if message_folder.lower().startswith('inbox.'):
                message_folder = message_folder[len('inbox.'):]

            subdir_path = posixpath.join(maildir_path, '.' + message_folder)
            if not os.path.isdir(subdir_path):
                os.mkdir(subdir_path, 0700)
                os.chown(subdir_path, uid, gid)

                for subdir_item in [self.DIR_CUR, self.DIR_NEW, self.DIR_TMP]:
                    os.mkdir(posixpath.join(subdir_path, subdir_item), 0700)
                    os.chown(posixpath.join(subdir_path, subdir_item), uid, gid)

                with open(posixpath.join(subdir_path, 'maildirfolder'), 'w'):
                    pass

                os.chown(posixpath.join(subdir_path, 'maildirfolder'), uid, gid)
                os.chmod(posixpath.join(subdir_path, 'maildirfolder'), 0600)

                with open(posixpath.join(maildir_path, 'subscriptions'), 'a+') as fp:
                    fp.write(message_folder + "\n")

            message_dir_path = posixpath.join(subdir_path, 'cur')

        flags_str = IMAPFlagsUtils.flags_to_maildir_string(message.flags)

        message_path = posixpath.join(message_dir_path, "%s%s" % (message.message_id, flags_str))
        logger.debug("Restore mail message to '%s'", message_path)

        with open(message_path, 'wb'):
            pass

        os.chown(message_path, uid, gid)
        os.chmod(message_path, 0600)

        with open(message_path, 'wb') as fp:
            fp.write(message.body)

    def _get_maildir_path(self, account):
        """Path to "Maildir" directory of specified mail account.

        :param plesk_mail_migrator.core.entities.mail_account.MailAccount account:
            Account to which message should be restored
        :rtype: str
        """
        return posixpath.join(
            self.get_parameter_value(self.MAILNAMES_DIR), account.domain, account.user_name, 'Maildir'
        )

    def _get_subscriptions_path(self, account):
        """Path to "subscriptions" file which lists IMAP folders

        :param plesk_mail_migrator.core.entities.mail_account.MailAccount account:
            Account to which message should be restored
        :rtype: str
        """
        return posixpath.join(self._get_maildir_path(account), 'subscriptions')

    def _get_folders(self, account):
        """Get list of IMAP folders for specified account by "subscriptions" file

        :param plesk_mail_migrator.core.entities.mail_account.MailAccount account:
            Account to which message should be restored
        :rtype: list[str]
        """
        try:
            with open(self._get_subscriptions_path(account)) as fp:
                return [
                    folder
                    for folder in [line.strip() for line in fp.read().split('\n')]
                    if folder != ''
                ]
        except IOError as e:
            if e.errno == errno.ENOENT:
                return []
            else:
                raise
