# Copyright 1999-2016. Parallels IP Holdings GmbH. All Rights Reserved.
package MySQLAgent::Backend::Shell;

use strict;
use MySQLAgent::Backend::Utils;
use File::Temp qw/tempdir/;
use POSIX;

use vars qw|@ISA|;

@ISA = qw|MySQLAgent::Backend::Base|;

# ============================ Public methods =================================

sub connect {
  my ($self) = @_;

  $self->{fakeHomeDir} = tempdir();
  $self->{configFile} = POSIX::tmpnam();

  if (open OPTSFILE, ">$self->{configFile}") {
      close OPTSFILE;
      chmod 0600, $self->{configFile};

      open OPTSFILE, ">$self->{configFile}";
      print OPTSFILE "[mysql]\npassword=";
      if (!$self->_mysql41OrNewer()) {
        print OPTSFILE $self->{password};
      } else {
        print OPTSFILE '"' . $self->{password} . '"';
      }
      close OPTSFILE;
  } else {
      $self->_set_errstr("Can't create temporary file");
  }

  my @version = MySQLAgent::Backend::Utils::getVersionArray();
  $self->{mysql_major_version} = @version[0];
  $self->{mysql_minor_version} = @version[1];
  $self->{mysql_build_version} = @version[2];

  my $version = MySQLAgent::Backend::Utils::getVersion();

  $self->{cmdline} = $self->_getCmd("mysql", "-B" . ($version && $version eq "3.23" ? "" : " -b"));

  if (MySQLAgent::Backend::Utils::doesSupportCharacterSets($self)) {
    $self->{sql_prefix} = MySQLAgent::Backend::Utils::getCharacterSetsSupportSql() . ";";
    $self->{sql_prefix} .= MySQLAgent::Backend::Utils::getNamesCharacterSetsSupportSql() . ";";
  }
}

sub disconnect {
  my ($self) = @_;

  rmdir $self->{fakeHomeDir};
  unlink $self->{configFile};
}

sub queryArrayOfHashes {
  my ($self, $sql, @params) = @_;

  my $resultSql = $self->_getSql($sql, @params);
  if (!defined($resultSql)) {
    $self->_set_errstr(sprintf("Unable to replace placeholders in SQL query: invalid parameters number (%d)\n%s", scalar(@params), $sql));
    return;
  }
  $sql = $resultSql;

  if (defined $self->{sql_prefix}) {
    $sql = $self->{sql_prefix} . $sql;
  }

  # quote ` from the shell and ' from the string
  $sql =~ s/'/'\\''/g;

  my @out = `echo '$sql' | $self->{cmdline}`;
  chomp @out;

  if ($#out > 0 and $out[0] =~ /^error/i) {
    $self->_set_errstr("Error: unable to execute $sql:\n" . (join "\n", @out));
    return;
  }

  my @data = map {[map {$_ eq 'NULL' ? undef : _unquote($_)} split (/\t/, $_, -1)]} @out;
  my $columns = shift @data;

  my @table;

  for (my $row = 0; $row < @data; $row++) {
    my $rowData = {};
    for (my $col = 0; $col < @$columns; $col++) {
        $rowData->{$columns->[$col]} = $data[$row][$col];
    }
    push @table, $rowData;
  }

  return \@table;
}

# =========================== Private methods =================================

sub _getCmd {
  my ($self, $cmd, $additionalOptions) = @_;

  if ($self->{password}) {
    $cmd .= " --defaults-file=" . $self->{configFile};
  }

  $cmd .= " --socket='$self->{socket}'" if $self->{socket};
  $cmd .= " -h '$self->{host}'" if $self->{host};
  $cmd .= " -u '$self->{user}'";
  $cmd .= " -P '$self->{port}'" if $self->{port};
  if ($self->{database} =~ /^-.*/) {
    $cmd .= " -- ";
  }
  $cmd .= " '$self->{database}'";

  # ~/.my.cnf is the top priority file for Linux and can interfere with
  # DB dumps, so we cheat here pretending HOME directory is empty
  # and there is no ~/.my.cnf
  $cmd = "HOME=\"$self->{fakeHomeDir}\" $cmd";

  return $cmd;
}

sub _mysql41OrNewer {
  my ($self) = @_;

  if (defined $self->{mysql_major_version} and defined $self->{mysql_minor_version}) {
    return $self->{mysql_major_version} > 4 || ($self->{mysql_major_version} == 4 and $self->{mysql_minor_version} >= 1);
  }
}

#
# unquotes \n \t \\ in the string
# used in parsing mysql batch output
#

sub _unquote {
  my ($str) = @_;

  my $len = length($str);
  my $res = "";
  for (my $i = 0; $i <= $len; ++$i) {
    my $s = substr($str, $i, 1);

    if ($i != $len and $s eq "\\") {
      my $s = substr($str, ++$i, 1);
      $res .= "\n" if $s eq "n";
      $res .= "\t" if $s eq "t";
      $res .= "\\" if $s eq "\\";
    } else {
      $res .= $s;
    }
  }
  return $res;
}

sub _getSql {
  my ($self, $sql, @params) = @_;
  my $paramsCount = scalar(@params);
  my $placeholdersCount = 0;

  $sql =~ s/\?/$self->_getSqlParam(\@params, \$placeholdersCount)/ge;

  if ($paramsCount != $placeholdersCount) {
    return undef;
  }
  return $sql;
}

sub _getSqlParam {
  my ($self, $params, $count) = @_;

  ++${$count};
  if (!@{$params}) {
     return '?';
  }
  my $param = shift(@{$params});
  if (defined($param)) {
    $param =~ s/'/''/g;
    return "'$param'";
  } else {
    return 'NULL';
  }
}

1;
