# Copyright 1999-2016. Parallels IP Holdings GmbH. All Rights Reserved.
package MySQLAgent::MySQLAgent;

use strict;

use JSON;

use MySQLAgent::Utils;
use MySQLAgent::Backend::DBI;
use MySQLAgent::Backend::Shell;

# Entry point
sub main {
    my $stdinData = MySQLAgent::Utils::readWholeStdin();
    my $inputData = decode_json($stdinData);
    my $dbh = _createDatabaseConnection($inputData->{'connection_settings'});
    my $result = $dbh->queryArrayOfHashes($inputData->{'query'}, @{$inputData->{'query_args'}});
    $dbh->disconnect();
    _exitWithRowSet($result);
}

# Create database connection to specified database
sub _createDatabaseConnection {
    my ($params) = @_;

    if ($params->{'backend'} eq 'dbi') {
        my $conn = MySQLAgent::Backend::DBI->new(%$params);
        return $conn if $conn->connect();
    } elsif ($params->{'backend'} eq 'shell') {
        my $conn = MySQLAgent::Backend::Shell->new(%$params);
        return $conn if $conn->connect();
    } elsif ($params->{'backend'} eq '') {
        # First, try to connect with DBI
        my $conn = MySQLAgent::Backend::DBI->new(%$params);
        return $conn if $conn->connect();

        # If connection with DBI failed, fallback to "mysql" utility usage
        my $conn = MySQLAgent::Backend::Shell->new(%$params);
        return $conn if $conn->connect();
    } else {
        _exitWithError(
            "Invalid backend type was specified. Allowed values: 'dbi', 'shell'."
        );
    }

    _exitWithError(
        "Can not connect to the database. Check that connection settings are specified correctly."
    );
}

# Exit from the script with error, print error as JSON
sub _exitWithError {
	my ($message) = @_;

	my $json = JSON->new->allow_nonref;
	print $json->pretty->encode({
		status => 'error',
		message => $message
	});
	exit 1;
}

# Exit from the script with success, print result row set as JSON
sub _exitWithRowSet {
	my ($rowset) = @_;

	my $json = JSON->new->allow_nonref;
	print $json->pretty->encode({
		status => 'ok',
		rowset => $rowset
	});
	exit 0;
}

1;