<?php

class Dispatcher {
    private $_logger;

    public function __construct() {
        ob_start();
        $this->_logger = Registry::get_instance()->set_logger(new Logger());
        register_shutdown_function(array($this, 'handle_shutdown'));
        set_exception_handler(array($this, 'handle_exception'));
        set_error_handler(array($this, 'handle_error'));
    }

    public function run() {
        $task_name = $_GET['task'];
        $data = json_decode(file_get_contents("php://input"), true);

        $this->_logger->info('Perform task ' . $task_name);

        $task_name_parts = array();
        foreach (explode('_', $task_name) as $part) {
            $task_name_parts[] = ucfirst($part);
        }
        $task_class_name = implode('', $task_name_parts) . 'Task';
        $task_instance = new $task_class_name($data);

        if (!$task_instance instanceof Task) {
            throw new Exception('Unknown task "' . $task_name . '"');
        }
        $this->_finalize($task_instance->run(), null);
    }

    public function handle_shutdown() {
        $error = error_get_last();
        if (is_null($error)) {
            return;
        }
        switch ($error['type']) {
            case E_ERROR:
            case E_PARSE:
            case E_COMPILE_ERROR:
            case E_CORE_ERROR:
                $error_message = 'PHP Fatal Error at line ' . $error['line'] . ' of file ' . $error['file'] . ': ' .
                    $error['message'];
                $this->_logger->error($error_message);
                $this->_finalize(null, $error_message);
        }
    }

    public function handle_exception(Exception $exception) {
        $error_message = 'Unhandled exception at line ' . $exception->getLine() . ' of file ' .
            $exception->getFile() . ': ' . $exception->getMessage();
        $this->_logger->error($error_message );
        $this->_logger->debug('Exception stacktrace:' . Logger::CRLF . $exception->getTraceAsString());
        $this->_finalize(null, $error_message);
    }

    public function handle_error($errno, $errstr, $errfile, $errline) {
        switch ($errno) {
            case E_WARNING:
                $this->_logger->warning(
                    'PHP Warning at line ' . $errline . ' of file ' . $errfile . ': ' . $errstr
                );
                break;
            case E_NOTICE:
                $this->_logger->debug(
                    'PHP Notice at line ' . $errline . ' of file ' . $errfile . ': ' . $errstr
                );
                break;
            default:
                $this->_logger->debug(
                    'Unknown PHP error ' . $errno . ' at line ' . $errline . ' of file ' . $errfile . ': ' . $errstr
                );
        }
        return true;
    }

    function _finalize($result, $error) {
        ob_end_clean();
        if (!is_null($error)) {
            http_response_code(500);
        }
        echo json_encode(array(
            'result' => $result,
            'error' => $error
        ));
        exit();
    }
}