<?php
// Copyright 1999-2017. Plesk International GmbH. All rights reserved.

/**
 * Abstract class which defines methods to return results of task execution to client.
 */
abstract class ResultEncoder
{
    /**
     * Start list.
     *
     * Call startListValue followed by addString, addInt, ..., startList, endList, ..., to put a value into the list.
     * Once all elements are added - call endList function.
     *
     * @return void
     */
    abstract public function startList();

    /**
     * End list. Call after startList function once all list elements are added.
     *
     * @return void
     */
    abstract public function endList();

    /**
     * Start dictionary.
     * Call first startDictionaryValue to add new dictionary key, then any function out of
     * addString, addInt, ..., startList, endList, ... to put a value into the dictionary key.
     *
     * Once all elements are added - call endDictionary function.
     *
     *
     * @return void
     */
    abstract public function startDictionary();

    /**
     * End dictionary. Call after startDictionary function once all dictionary elements are added.
     *
     * @return void
     */
    abstract public function endDictionary();

    /**
     * Add string to results.
     * Use within list or dictionary or on its own.
     *
     * @param string $str
     * @return mixed
     */
    abstract public function addString($str);

    /**
     * Add integer value to results.
     *
     * @param int $int
     * @return void
     */
    abstract public function addInt($int);

    /**
     * Add boolean value to results.
     *
     * @param bool $b
     * @return void
     */
    abstract public function addBoolean($b);

    /**
     * Add null value to results.
     * @return void
     */
    abstract public function addNull();

    /**
     * Add raw (= already encoded) value to the resulting
     * @param $raw
     * @return void
     */
    abstract public function addRawValue($raw);

    /**
     * Start dictionary value with specified key. Call addString, addBoolean, etc. after this function to add value.
     *
     * @param string $key
     * @return void
     */
    abstract public function startDictionaryValue($key);

    /**
     * Start new list item. Call addString, addBoolean, etc., after this function to actually add an element.
     *
     * @return void
     */
    abstract public function startListValue();

    /**
     * Get result of task execution as string. The string should be passed back to client in HTTP response.
     *
     * @return string
     */
    abstract public function getData();

    /**
     * The most simple way to return PHP variable as task execution result.
     * For example, when returning result as JSON,
     * PHP string is encoded as JSON string, boolean is encoded as PHP boolean,
     * PHP array is encoded as JSON dictionary, etc.
     *
     * @param $value
     */
    public function addPhpValue($value)
    {
        if ($value === null) {
            $this->addNull();
        } elseif (is_bool($value)) {
            $this->addBoolean($value);
        } elseif (is_int($value)) {
            $this->addInt($value);
        } elseif (is_string($value)) {
            $this->addString($value);
        } elseif (is_array($value)) {
            $this->startDictionary();
            foreach ($value as $k => $v) {
                $this->startDictionaryValue($k);
                $this->addPhpValue($v);
            }
            $this->endDictionary();
        }
    }
}